<?php

namespace Mtc\ContentManager;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use JetBrains\PhpStorm\ArrayShape;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Contracts\ElementRepositoryContract;
use Mtc\ContentManager\Models\Content;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Menu;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Traits\EnsuresSlug;

class ElementRepository implements ElementRepositoryContract
{
    use EnsuresSlug;

    public function __construct(
        protected ContentElement $elementModel
    ) {
        //
    }

    /**
     * Attach model to repository
     *
     * @param ContentElement $element
     * @return $this
     */
    public function setModel(ContentElement $element): self
    {
        $this->elementModel = $element;
        return $this;
    }

    /**
     * Find the element
     *
     * @param int $id
     * @return ContentElement|Model
     */
    public function find(int $id): ContentElement
    {
        return $this->elementModel->newQuery()->findOrFail($id);
    }

    /**
     * Load element by id for usage
     *
     * @param int $id
     * @return $this
     */
    public function load(int $id): self
    {
        $this->elementModel = $this->elementModel->newQuery()->findOrFail($id);
        return $this;
    }

    /**
     * Get list
     *
     * @param string|null $searchTerm
     * @param string|null $sortBy
     * @param int $limit
     * @param bool $active
     * @param bool $paginate
     * @param bool $drag_and_drop
     * @return LengthAwarePaginator|Collection
     */
    public function getList(
        string $searchTerm = null,
        string $sortBy = null,
        int $limit = 10,
        bool $active = true,
        bool $paginate = true,
        bool $drag_and_drop = true
    ): LengthAwarePaginator|Collection {
        $query = $this->elementModel->newQuery()
            ->with('fields.childElement.fields.childElement.fields.childElement.fields')
            ->when(!empty($searchTerm), fn($query) => $query->searchTerm($searchTerm))
            ->when($active, fn($query) => $query->where('is_enabled', 1))
            ->when($drag_and_drop, fn($query) => $query->where('drag_and_drop', 1))
            ->setSortBy($sortBy)
            ->latest();

        return $paginate
            ? $query->paginate($limit)
            : $query->take($limit)->get();
    }

    /**
     * Create a new content element
     *
     * @param string $title
     * @param string|null $subtitle
     * @param string|null $icon
     * @param string|null $category
     * @param bool $enabled
     * @return ContentElement|Model
     */
    public function create(
        string $title,
        string $subtitle = null,
        string $icon = null,
        string $category = null,
        bool $enabled = true,
    ): ContentElement {
        return $this->elementModel->newQuery()
            ->create([
                'title' => $title,
                'slug' => $this->ensureSlug($title, $this->elementModel),
                'subtitle' => $subtitle,
                'icon' => $icon,
                'category' => $category,
                'is_enabled' => $enabled,
            ]);
    }

    public function update(ContentElement $element, array $input): ContentElement
    {
        $this->setModel($element);
        $element->fill([
            'title' => $input['title'],
            'subtitle' => $input['subtitle'] ?? '',
            'icon' => $input['icon'] ?? null,
            'category' => $input['category'] ?? null,
            'is_enabled' => $input['is_enabled'] ?? false,
        ])->save();

        $this->syncFields(collect($input['fields'] ?? []));
        return $element;
    }

    public function makeCopy(int $elementId, string $title): ContentElement
    {
        $original = $this->find($elementId);

        $this->elementModel = $this->create(
            title: $title,
            subtitle: $original->subtitle,
            icon: $original->icon,
            category: $original->category,
            enabled: false
        );

        $original->fields
            ->each(fn (ContentElementField $field) => $this->addField([
                'child_element_id' => $field->child_element_id,
                'field_type' => $field->field_type,
                'name' => $field->name,
                'data' => $field->data,
                'meta' => $field->meta,
            ]));

        return $this->elementModel;
    }

    /**
     * Add field to content element
     *
     * @param array $field
     * @return $this
     */
    #[ArrayShape([
        'child_element_id' => "int",
        'field_type' => "string",
        'name' => "string",
        'data' => "array",
        'choices' => "array",
    ])]
    public function addField(array $field): self
    {
        $this->elementModel->fields()
            ->create([
                'child_element_id' => $field['child_element_id'] ?? null,
                'field_type' => !empty($field['child_element_id']) ? 'element' : $field['field_type'],
                'name' => $field['name'],
                'slug' => $field['slug'] ?? Str::slug($field['name']),
                'data' => $field['data'] ?? [],
                'meta' => $field['meta'] ?? [],
                'order' => $field['order'] ?? 0,
            ]);

        return $this;
    }

    /**
     * Remove a content element
     *
     * @param int $elementId
     * @return bool
     */
    public function destroy(int $elementId): bool
    {
        /** @var ContentElement $element */
        $element = $this->elementModel
            ->newQuery()
            ->find($elementId);

        $element->fields()->delete();
        return $element->delete();
    }

    /**
     * Get the list of places where content element is used
     *
     * @param int $elementId
     * @return array
     */
    public function getUsage(int $elementId): array
    {
        return [
            'templates' => $this->findTemplatesUsingElement($elementId),
            'freeform_pages' => $this->findPagesUsingElement($elementId),
            'global_elements' => $this->findGlobalElementsUsingElement($elementId),
            'menus' => $this->findMenusUsingElement($elementId),
        ];
    }

    /**
     * Show full usage list of a specific content type
     *
     * @param int $elementId
     * @param string $type
     * @return Collection
     */
    public function fullUsageList(int $elementId, string $type): Collection
    {
        return match ($type) {
            'templates' => $this->findTemplatesUsingElement($elementId, 0),
            'freeform_pages' => $this->findPagesUsingElement($elementId, 0),
            'global_elements' => $this->findGlobalElementsUsingElement($elementId, 0),
            'menus' => $this->findMenusUsingElement($elementId, 0),
        };
    }

    /**
     * Find Template records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findTemplatesUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Template::query()
            ->whereHas('elements', fn($query) => $query->where('element_id', $elementId))
            ->withActivePageCount()
            ->withInactivePageCount()
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Find free-form pages (no template) records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findPagesUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Page::query()
            ->whereNull('template_id')
            ->whereHas('content', fn($query) => $query->where('element_id', $elementId))
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findGlobalElementsUsingElement(int $elementId, int $limit = 10): Collection
    {
        return GlobalContent::query()
            ->where('element_id', $elementId)
            ->withTemplateCount()
            ->withFreeformPageCount()
            ->limit($limit ?: null)
            ->get();
    }


    /**
     * Find global content element records that use content element
     *
     * @param int $elementId
     * @param int $limit
     * @return Collection
     */
    public function findMenusUsingElement(int $elementId, int $limit = 10): Collection
    {
        return Menu::query()
            ->whereHas('entries', fn ($query) => $query->where('element_id', $elementId))
            ->limit($limit ?: null)
            ->get();
    }

    /**
     * Sync fields of a content element
     *
     * @param Collection $fields
     * @return void
     */
    protected function syncFields(Collection $fields)
    {
        $fields = $fields->map(function ($field, $index) {
            $field['order'] = $index;
            if (empty($field['child_element_id']) && !empty($field['element_id'])) {
                $field['child_element_id'] = $field['element_id'] ?? null;
            }
            $field['data'] = $this->fieldDataValues($field);
            return $field;
        });

        $requestEntryIds = $fields->pluck('id');
        $currentFields = $this->elementModel->fields()->get();

        $currentFields->reject(fn($field) => in_array($field->id, $requestEntryIds->toArray()))
            ->each(fn($field) => $field->delete());

        $currentFields->filter(fn($field) => in_array($field->id, $requestEntryIds->toArray()))
            ->each(fn($field) => $field->fill($fields->where('id', $field->id)->first())->save());

        $fields->reject(fn($field) => is_numeric($field['id'] ?? ''))
            ->each(fn($field) => $this->addField($field));
    }

    protected function fieldDataValues($field): array
    {
        return collect($field)->except([
            'meta',
            'data',
            'id',
            'name',
            'content',
            'children',
            'field_type',
            'order',
            'element_id',
            'global_content_id',
        ])->toArray();
    }
}
