<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\PageVersionFactory;
use Mtc\ContentManager\Models\Comment;
use Ramsey\Uuid\Uuid;

abstract class VersionModel extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'page_versions';

    /**
     * @var string[]
     */
    protected $fillable = [
        'page_id',
        'author_id',
        'is_active',
        'structure',
        'pending_review',
        'request_changes',
    ];

    /**
     * @var string[]
     */
    protected $casts = [
        'is_active' => 'bool',
        'pending_review' => 'bool',
        'request_changes' => 'bool',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        self::creating(fn (self $pageVersion) => $pageVersion->ensureUuid());
    }

    /**
     * Model factory
     *
     * @return PageVersionFactory
     */
    protected static function newFactory()
    {
        return PageVersionFactory::new();
    }

    /**
     * Relationship to the owner page of this version
     *
     * @return BelongsTo
     */
    public function page(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.page_model'));
    }

    /**
     * Author of this page version
     *
     * @return BelongsTo
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(Config::get('auth.providers.users.model'));
    }

    /**
     * Relationship with content elements on version
     *
     * @return MorphMany
     */
    public function content(): MorphMany
    {
        return $this->morphMany(Config::get('pages.version_content_model'), 'owner')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content elements on version
     *
     * @return MorphMany
     */
    public function allContent(): MorphMany
    {
        return $this->morphMany(Config::get('pages.version_content_model'), 'owner')
            ->orderBy('order');
    }

    public function scopeWithCommentCount(Builder $query): void
    {
        $query->addSelect([
            'comment_count' => Comment::query()
                ->selectRaw('COUNT(*)')
                ->where('commentable_type', $this->getMorphClass())
                ->whereColumn('commentable_id', $this->getTable() . '.id')
                ->whereNull('resolved_at')
        ]);
    }

    public function url(): string
    {
        return '';
    }

    private function ensureUuid()
    {
        if (empty($this->attributes['uuid'])) {
            $this->uuid = Uuid::uuid4();
        }
    }
}
