<?php

namespace Mtc\ContentManager;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;
use Mtc\ContentManager\Contracts\Media;
use Mtc\ContentManager\Contracts\MediaTag;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Http\Requests\MediaResizeRequest;
use Mtc\ContentManager\Http\Requests\MediaUploadRequest;
use Mtc\ContentManager\Jobs\GenerateSizeForMediaFileJob;
use Mtc\ContentManager\Jobs\RegenerateSizesForMediaFileJob;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class MediaRepository
{
    use DispatchesJobs;

    public function __construct(
        protected Media $model,
        protected MediaUse $mediaUse,
        protected MediaTag $tag,
    ) {
        //
    }

    /**
     * List of media files
     *
     * @param Request $request
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function getList(Request $request): \Illuminate\Contracts\Pagination\LengthAwarePaginator
    {
        return $this->model->newQuery()
            ->when($request->has('model'), fn($query) => $query->where('parent_type', $request->input('model')))
            ->when($request->has('model_id'), fn($query) => $query->where('parent_id', $request->input('model_id')))
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->setFilters($request->input('filters', []))
            ->paginate($request->input('per_page'));
    }

    /**
     * Find media file
     *
     * @param int $id
     * @return Media|Model
     */
    public function find(int $id): Media
    {
        return $this->model->newQuery()
            ->findOrFail($id);
    }

    /**
     * Import image from an external URL
     *
     * @param string $url
     * @param string $model
     * @return Media
     * @throws \Exception
     */
    public function importImageFromUrl(string $url, string $model = ''): Media
    {
        $uploadPath = $this->storagePrefix() . ltrim($model . '/' . date('Y-M'), '/');
        $fileName = Carbon::now()->format('U') . pathinfo($url, PATHINFO_BASENAME);
        $image = (new ImageManager(Config::get('media.image_manager_config', [])))
            ->make(file_get_contents($url));

        Storage::disk(Config::get('filesystems.default_media'))
            ->put(
                ltrim($uploadPath, '/') . '/' . $fileName,
                $image->stream(),
                ['visibility' => 'public']
            );

        return $this->createMediaRecordForFile($uploadPath, $fileName, $image->mime(), true);
    }


    /**
     * Handle new image upload
     *
     * @param Request $request
     * @param Media|null $mediaToUpdate
     * @return Media
     * @throws \Exception
     */
    public function upload(MediaUploadRequest $request, Media $mediaToUpdate = null): Media
    {
        return $request->fileIsImage()
            ? $this->uploadImage($request, $mediaToUpdate)
            : $this->uploadFile($request, $mediaToUpdate);
    }

    /**
     * Handle new image upload
     *
     * @param Request $request
     * @return Media
     * @throws \Exception
     */
    public function uploadImage(MediaUploadRequest $request, Media $mediaToUpdate = null): Media
    {
        $uploadPath = $this->storagePrefix() . ltrim($request->input('model', '') . '/' . date('Y-M'), '/');
        $fileName = Carbon::now()->format('U') . $request->file('file')->getClientOriginalName();

        $image = (new ImageManager(Config::get('media.image_manager_config', [])))
            ->make($request->file('file')->getPathname())
            ->orientate();

        Storage::disk(config('filesystems.default_media'))
            ->put(
                $uploadPath . '/' . $fileName,
                $image->stream(),
                ['visibility' => 'public']
            );

        return $mediaToUpdate
            ? $this->updateRecordWithFile(
                $mediaToUpdate,
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                true,
                $request->user()?->id
            )
            : $this->createMediaRecordForFile(
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                true,
                $request->user()?->id
            );
    }

    /**
     * Handle new file upload
     *
     * @param Request $request
     * @return Media
     * @throws \Exception
     */
    public function uploadFile(MediaUploadRequest $request, Media $mediaToUpdate = null): Media
    {
        $uploadPath = ltrim($request->input('model') . '/' . date('Y-M'), '/');
        $fileName = Carbon::now()->format('U') . $request->file('file')->getClientOriginalName();

        Storage::disk(config('filesystems.default_media'))
            ->putFileAs($uploadPath, $request->file('file'), $fileName);

        return $mediaToUpdate
            ? $this->updateRecordWithFile(
                $mediaToUpdate,
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                false,
                $request->user()?->id,
            )
            : $this->createMediaRecordForFile(
                $uploadPath,
                $fileName,
                $request->file('file')->getMimeType(),
                false,
                $request->user()?->id,
            );
    }

    /**
     * Attach media use cases to model
     *
     * @param int[] $media
     * @param Model $model
     * @param array $meta
     * @return void
     */
    public function setUsesForModel(array $mediaIds, Model $model, array $meta = [])
    {
        $allowed_sizes = $meta['allowed_sizes'] ?? [];
        if (!empty($model->default_allowed_media_sizes)) {
            $allowed_sizes = array_merge($allowed_sizes, $model->default_allowed_media_sizes);
        }
        $this->model->newQuery()
            ->whereIn('id', $mediaIds)
            ->get()
            ->each(fn(Media $media) => $media->update([
                // only set if no meta already exists
                'alt_text' => $media->getAttribute('alt_text') ?? $meta['alt_text'] ?? null,
                'title' => $media->getAttribute('title') ?? $meta['title'] ?? null,
                'caption' => $media->getAttribute('caption') ?? $meta['caption'] ?? null,
                'description' => $media->getAttribute('description') ?? $meta['description'] ?? null,
            ]))
            ->each(fn(Media $media, $index) => $media->uses()
                ->updateOrCreate([
                    'owner_type' => $model->getMorphClass(),
                    'owner_id' => $model->id,
                ], [
                    // update meta from passed with fallback of media object default
                    'alt_text' => $meta['alt_text'] ?? $media->getAttribute('alt_text'),
                    'title' => $meta['title'] ?? $media->getAttribute('title'),
                    'caption' => $meta['caption'] ?? $media->getAttribute('caption'),
                    'description' => $meta['description'] ?? $media->getAttribute('description'),
                    'dimensions' => $meta['dimensions'] ?? null,
                    'primary' => $index == 0 && ($meta['primary'] ?? null),
                    'secondary' => $meta['secondary'] ?? null,
                    'allowed_sizes' => $allowed_sizes,
                ]))
                ->each(function (Media $media) use ($allowed_sizes) {
                    foreach ($allowed_sizes as $dimensions) {
                        $size = ImageSize::fromArray($this->pathStringToArray($dimensions));
                        if ($this->sizeExists($media, $size) === false) {
                            $this->dispatch(new GenerateSizeForMediaFileJob($media, $size));
                        }
                    }
                });
    }

    /**
     * Update media use order for model
     *
     * @param int[] $media
     * @param Model $model
     * @return void
     */
    public function setUseOrdering(array $mediaOrder, Model $model)
    {
        $data = collect($mediaOrder)
            ->map(fn($order, $id) => [
                'media_id' => $id,
                'order' => $order,
                'owner_id' => $model->id,
                'owner_type' => $model->getMorphClass()
            ])
            ->toArray();
        $this->mediaUse->newQuery()
            ->upsert($data, ['media_id', 'owner_id', 'owner_type'], ['order']);
    }

    /**
     * Get list of files in library
     *
     * @param Request $request
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator
     */
    public function getFiles(Request $request)
    {
        return $this->model->newQuery()
            ->when($request->input('type'), fn($query) => $query->where('type', $request->input('type')))
            ->when($request->input('tag'), function ($query) use ($request) {
                return $query->whereHas('tags', fn($tag_query) => $tag_query->where('tag', $request->input('tag')));
            })
            ->when($request->input('upload_date'), function ($query) use ($request) {
                return $query->where('upload_date', $request->input('upload_date'));
            })
            ->paginate(Config::get('media.library_page_size'));
    }

    public function getUploadDates()
    {
        return $this->model->newQuery()
            ->select('upload_date')
            ->distinct()
            ->orderByDesc('upload_date')
            ->pluck('upload_date', 'upload_date')
            ->map(fn($upload_date) => Carbon::createFromFormat('Y-m', $upload_date)->format('F Y'));
    }

    public function getTypes()
    {
        return array_keys(Config::get('media.file_types', []));
    }

    /**
     * Get list of tags used
     *
     * @return Collection
     */
    public function getTags()
    {
        return $this->tag->newQuery()
            ->distinct()
            ->select('tag')
            ->get();
    }

    /**
     * Coordinates are defined as x1, y1, x2, y2
     *
     * @param MediaResizeRequest $request
     * @param array $coordinates
     * @return string
     * @throws \Exception
     */
    public function resize(MediaResizeRequest $request, array $coordinates): string
    {
        $cropWidth = $coordinates[2] - $coordinates[0];
        $cropHeight = $coordinates[3] - $coordinates[1];
        $size = new ImageSize($request->width(), $request->height());
        $original = $request->media()->media;

        $file = Storage::disk(Config::get('filesystems.default_media'))->get($original->getOriginalFilePath());
        $image = (new ImageManager(Config::get('media.image_manager_config', [])))
            ->make($file)
            ->crop($cropWidth, $cropHeight, $coordinates[0], $coordinates[1])
            ->resize($size->getWidth(), $size->getHeight())
            ->fit($size->getWidth(), $size->getHeight());

        Storage::disk(Config::get('filesystems.default_media'))
            ->put($size->fileDestination($original->src), $image->stream());

        return Storage::disk(Config::get('filesystems.default_media'))->url($size->pathOnDisk($original->src));
    }

    /**
     * Remove a size
     *
     * @param MediaUse $mediaUse
     * @param int $width
     * @param int $height
     * @return void
     */
    public function removeMediaUseSize(MediaUse $mediaUse, int $width, int $height): void
    {
        $size = new ImageSize($width, $height);
        Storage::disk(Config::get('filesystems.default_media'))->url($size->pathOnDisk($mediaUse->seo_url));
    }

    /**
     * Remove media file
     *
     * @param int|array $mediaId
     */
    public function destroyMedia(int|array $mediaId)
    {
        if (is_int($mediaId)) {
            $mediaId = [$mediaId];
        }

        $this->model->newQuery()
            ->with('uses')
            ->whereIn('id', $mediaId)
            ->get()
            ->each(function (Media $media) {
                $this->purgeFiles($media);
                $media->uses()->delete();
                $media->delete();
            });
    }


    /**
     * Remove all files for allowed dimensions of this media file that are defined by this usage
     *
     * @param MediaUse $mediaUse
     * @return bool
     */
    public function removeMediaUseFiles(MediaUse $mediaUse): bool
    {
        $sizes_used_on_other_media_uses = $mediaUse
            ->media
            ->uses()->where('id', '!=', $mediaUse->id)->get()
            ->pluck('allowed_sizes')
            ->flatten();

        return collect($mediaUse->allowed_sizes)
            ->filter(fn($path) => $this->pathIsWidthAndHeight($path))
            ->filter(fn($path) => !in_array($path, $sizes_used_on_other_media_uses))
            ->map(fn($dimensions) => ImageSize::fromArray($this->pathStringToArray($dimensions)))
            ->reject(fn(ImageSize $size) => $this->removeFile($size->pathOnDisk($mediaUse->media->src)))
            ->isEmpty();
    }

    /**
     * Check if the path does represent width and height
     *
     * @param string $path
     * @return bool
     */
    public function pathIsWidthAndHeight(string $path): bool
    {
        return preg_match('/^([1-9][0-9]{0,3})x([1-9][0-9]{0,3})$/', $path);
    }

    /**
     * Convert dimension string to an array
     *
     * @param string $dimensions
     * @return array
     */
    public function pathStringToArray(string $dimensions): array
    {
        list($width, $height) = explode('x', $dimensions);
        return [
            'width' => (int)$width,
            'height' => (int)$height
        ];
    }

    /**
     * Get the prefix for storage path
     * (in case system needs to modify this - e.g. tenancy)
     *
     * @return string
     */
    public function storagePrefix(): string
    {
        return '';
    }

    /**
     * Generate a new size for the uploaded media file
     *
     * @param Media $media
     * @param ImageSize $size
     * @return mixed|void
     * @throws \Exception
     */
    public function createSize(Media $media, ImageSize $size, bool $overwrite = false)
    {
        if ($overwrite === false && $this->sizeExists($media, $size)) {
            return;
        }

        return $media->resizableMime()
            ? $this->makeSizeWithDimensions($media, $size)
            : $this->copyToSizePath($media, $size);
    }

    /**
     * Create Media record and size for an uploaded file
     *
     * @param string $uploadPath
     * @param string $fileName
     * @param string $mime
     * @param bool $image
     * @param ?int $creator
     * @return Media
     * @throws \Exception
     */
    protected function createMediaRecordForFile(
        string $uploadPath,
        string $fileName,
        string $mime,
        bool $image = false,
        ?int $creator = null
    ): Media {
        /** @var Media $media */
        $media = $this->model->newQuery()
            ->create([
                'path' => $uploadPath,
                'src' => $fileName,
                'type' => $this->determineFileType($mime),
                'uploaded_by' => $creator,
            ]);

        if ($image) {
            $this->createSize($media, ImageSize::fromArray(Config::get('media.default_thumbnail_size', [])));
        }
        return $media;
    }

    /**
     * Update Media record and size for an uploaded file
     *
     * @param Media $media
     * @param string $uploadPath
     * @param string $fileName
     * @param string $mime
     * @param bool $image
     * @param ?int $creator
     * @return Media
     * @throws \Exception
     */
    protected function updateRecordWithFile(
        Media $media,
        string $uploadPath,
        string $fileName,
        string $mime,
        bool $image = false,
        ?int $creator = null
    ): Media {
        $media->update([
            'path' => $uploadPath,
            'src' => $fileName,
            'type' => $this->determineFileType($mime),
            'uploaded_by' => $creator,
        ]);

        if ($image) {
            $this->createSize($media, ImageSize::fromArray(Config::get('media.default_thumbnail_size', [])), true);
            $this->dispatch(new RegenerateSizesForMediaFileJob($media));
        }
        return $media;
    }

    /**
     * Purge files before removing media record
     *
     * @param Media $media
     * @return void
     */
    protected function purgeFiles(Media $media): void
    {
        $media->uses->each(function (MediaUse $mediaUse) {
            $this->removeMediaUseFiles($mediaUse);
            $mediaUse->delete();
        });

        collect($media->uses->pluck('dimension')->flatten(1))
            ->prepend($media->path . '/' . $media->src)
            ->prepend($media->thumbnailFilePath())
            ->each(fn($file) => $this->removeFile($file));
    }

    /**
     * Remove file from storage
     *
     * @param $path
     * @return bool
     */
    protected function removeFile($path): bool
    {
        return Storage::disk(Config::get('filesystems.default_media'))->delete($path);
    }

    /**
     * Determine file type from mime
     *
     * @param string $mime
     * @return string
     */
    protected function determineFileType(string $mime): string
    {
        return collect(config('media.media_types'))
                ->filter(fn($section) => in_array($mime, $section))
                ->map(fn($mies, $name) => $name)
                ->first() ?? 'other';
    }

    /**
     * Check if size exists
     *
     * @param Media $media
     * @param ImageSize $size
     * @return bool
     * @throws \Exception
     */
    protected function sizeExists(Media $media, ImageSize $size): bool
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->exists($size->fileDestination($media->src));
    }

    /**
     * Create a size of an image based on dimensions
     *
     * @param Media $media
     * @param ImageSize $size
     * @return mixed
     * @throws \Exception
     */
    protected function makeSizeWithDimensions(Media $media, ImageSize $size)
    {
        $original = Storage::disk(Config::get('filesystems.default_media'))->get($media->getOriginalFilePath());
        $image = (new ImageManager(Config::get('media.image_manager_config', [])))
            ->make($original)
            ->fit($size->getWidth(), $size->getHeight());

        Storage::disk(Config::get('filesystems.default_media'))
            ->put(
                $size->pathOnDisk($media->src),
                $image->stream(),
                ['visibility' => 'public']
            );

        return $image->response();
    }

    /**
     * Copy file to the relevant size path
     *
     * @param Media $media
     * @param ImageSize $size
     * @return bool
     */
    protected function copyToSizePath(Media $media, ImageSize $size)
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->copy($media->path . $media->src, $size->pathOnDisk($media->src));
    }

    /**
     * Find requested file details - dimensions, type, path
     *
     * @param  $link_to_file
     * @return array|string[]
     */
    protected function getRequestedFileDetails($link_to_file): array
    {
        $file_details = pathinfo($link_to_file);
        $dimensions = pathinfo($file_details['dirname'], PATHINFO_BASENAME);

        // No extension means this is not a file access
        if (empty($file_details['extension'])) {
            throw new NotFoundHttpException('Resource is not a file', null, 404);
        }

        if ($this->pathIsWidthAndHeight($dimensions) == false) {
            return [
                'type' => 'original',
            ];
        }

        $dimensions = explode('x', $dimensions);
        return [
            'type' => 'size',
            'width' => $dimensions[0],
            'height' => $dimensions[1],
            'file_name' => $file_details['basename'],
        ];
    }

    /**
     * Check if dimensions are allowed for the media use
     *
     * @param MediaUse $mediaUse
     * @param $dimensions
     * @return bool
     */
    protected function areDimensionsAllowed(MediaUse $mediaUse, $dimensions): bool
    {
        return in_array($dimensions['width'] . 'x' . $dimensions['height'], $mediaUse->allowed_sizes);
    }
}
