<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Models\Media as MediaModel;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Tests\TestCase;

class MediaRepositoryTest extends TestCase
{
    use RefreshDatabase;
    use WithFaker;

    public function testCorrectPathForPathCheck()
    {
        $valid_strings = [
            '100x100',
            '2000x1000',
        ];

        foreach ($valid_strings as $valid_string) {
            self::assertTrue(Media::pathIsWidthAndHeight($valid_string));
        }
    }

    public function testInccorrectPathForPathCheck()
    {
        $tests = [
            '10000x10000',
            '100x10x0',
            'aaa',
            'd321=1as=dpa-d',
            '100x',
            '100x0',
            '0x0',
            'e0x0',
        ];

        foreach ($tests as $test_string) {
            self::assertFalse(Media::pathIsWidthAndHeight($test_string));
        }
    }

    public function testSetUsesForModel()
    {
        $page = Page::factory()->create();
        $medias = MediaModel::factory(5)->create();

        Media::setUsesForModel($medias->pluck('id')->toArray(), $page);

        $query = MediaUse::query()
            ->where('owner_id', $page->id)
            ->where('owner_type', $page->getMorphClass());

        $this->assertEquals(5, $query->count());
    }

    public function testSetUseMetaForModel()
    {
        $page = Page::factory()->create();
        $medias = MediaModel::factory(2)->create();

        $meta = [
            'dimensions' => 'foo-baz',
            'allowed_sizes' => [
                '100x100',
                '200x200'
            ],
            'title' => 'foo',
            'caption' => 'baz',
            'description' => 'bar',
            'alt_text' => 'lorem-ipsum'
        ];
        Media::setUsesForModel($medias->pluck('id')->toArray(), $page, $meta);

        $model = MediaUse::query()
            ->where('owner_id', $page->id)
            ->where('owner_type', $page->getMorphClass())
            ->first();

        $this->assertEquals($meta['dimensions'], $model->dimensions);
        $this->assertEquals($meta['allowed_sizes'], $model->allowed_sizes);
        $this->assertEquals($meta['title'], $model->title);
        $this->assertEquals($meta['caption'], $model->caption);
        $this->assertEquals($meta['description'], $model->description);
        $this->assertEquals($meta['alt_text'], $model->alt_text);
    }

    public function testSetUseOrdering()
    {
        $page = Page::factory()->create();
        $medias = MediaModel::factory(5)->create();

        Media::setUsesForModel($medias->pluck('id')->toArray(), $page);

        $ordering = $medias->keyBy('id')
            ->map(fn () => random_int(0, 1000));

        Media::setUseOrdering($ordering->toArray(), $page);

        MediaUse::query()
            ->where('owner_id', $page->id)
            ->where('owner_type', $page->getMorphClass())
            ->get()
            ->each(fn (MediaUse $use) => $this->assertEquals($ordering[$use->media_id], $use->order));
    }

    public function testImportImageFromUrl()
    {
        Storage::fake('media');
        $imageUrl = $this->faker->image();
        if (empty($imageUrl)) {
            $this->markTestSkipped('Faker unable to generate image');
        }
        $media = Media::importImageFromUrl($imageUrl);

        $this->assertTrue($media->exists);
        Storage::disk('media')->assertExists($media->getOriginalFilePath());
    }
}
