<?php

namespace Mtc\ContentManager\Contracts;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\CommentFactory;

/**
 * @property int $id
 * @property string $commentable_type
 * @property int $commentable_id
 * @propery int $parent_id
 * @property int $author_id
 * @property null|string $content
 * @property null|Carbon $resolved_at
 * @property null|Carbon $created_at
 * @property null|Carbon $updated_at
 */
abstract class Comment extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'comments';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'commentable_type',
        'commentable_id',
        'parent_id',
        'author_id',
        'content',
        'resolved_at',
    ];

    protected $casts = [
        'resolved_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    /**
     * Add attributes to json response
     *
     * @var string[]
     */
    protected $appends = [
        'time_posted',
    ];

    /**
     * Model factory
     *
     * @return CommentFactory
     */
    protected static function newFactory()
    {
        return CommentFactory::new();
    }

    /**
     * Relationship with parent comment
     *
     * @return BelongsTo
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(static::class, 'parent_id');
    }

    /**
     * Relationship with child comments
     *
     * @return HasMany
     */
    public function children(): HasMany
    {
        return $this->hasMany(static::class, 'parent_id');
    }

    /**
     * Relationship with author of the comment
     *
     * @return BelongsTo
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'), 'author_id');
    }

    /**
     * Relationship with users tagging in comments
     *
     * @return BelongsToMany
     */
    public function taggedUsers(): BelongsToMany
    {
        return $this->belongsToMany(config('auth.providers.users.model'), 'comment_tagged_users');
    }

    /**
     * Relationship with the object to which this comment is attached to (page/template)
     *
     * @return MorphTo
     */
    public function commentable(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * time_posted as a formatted value
     *
     * @return string
     */
    public function getTimePostedAttribute(): string
    {
        if ($this->created_at->isToday()) {
            $day_text = 'Today';
        } elseif ($this->created_at->isYesterday()) {
            $day_text = 'Yesterday';
        } else {
            $day_text = $this->created_at->format('d/M/y');
        }
        return $this->created_at->format('g:i A ') . $day_text;
    }

    /**
     * Check if comment is resolved
     *
     * @return bool
     */
    public function isResolved(): bool
    {
        return !empty($this->resolved_at);
    }
}
