<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\MediaSize;
use Mtc\ContentManager\Seeds\DefaultMediaSizeSeeder;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class MediaSizeControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();

        MediaSize::factory(5)->create();
        $response = $this->getJson(route('content.media-sizes.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, count($response->json()));
    }

    public function testStore()
    {
        $this->asUser();

        $test_a = [
            'width' => 100,
            'height' => 100,
            'label' => 'foo'
        ];
        $test_b = [
            'width' => 400,
            'height' => 300,
            'label' => 'baz'
        ];
        $response = $this->postJson(route('content.media-sizes.store'), [
            'sizes' => "100x100:foo\n400x300:baz",
        ]);

        $response->assertStatus(200);
        $this->assertEquals(2, count($response->json()));
        $this->assertTrue(MediaSize::query()->where($test_a)->exists());
        $this->assertTrue(MediaSize::query()->where($test_b)->exists());

    }

    public function testForDimensions()
    {
        $this->asUser();

        MediaSize::factory(4)->create([
            'width' => 120,
            'height' => 300
        ]);
        MediaSize::factory(1)->create([
            'width' => 100,
            'height' => 100
        ]);

        $response = $this->postJson(route('content.media-sizes.for-dimensions'), [
            'sizes' => ['120x300']
        ]);
        $response->assertStatus(200);
        $this->assertEquals(4, count($response->json()));

        $response = $this->postJson(route('content.media-sizes.for-dimensions'), [
            'sizes' => ['100x100']
        ]);
        $response->assertStatus(200);
        $this->assertEquals(1, count($response->json()));

        $response = $this->postJson(route('content.media-sizes.for-dimensions'), [
            'sizes' => [
                '120x300',
                '100x100',
            ]
        ]);
        $response->assertStatus(200);
        $this->assertEquals(5, count($response->json()));
    }

    public function testDestroy()
    {
        $this->asUser();

        $size = MediaSize::factory()->create();
        $response = $this->deleteJson(route('content.media-sizes.destroy', $size));

        $response->assertStatus(200);
        $this->assertFalse(MediaSize::query()->where('id', $size->id)->exists());
    }

    public function testSeed()
    {
        $this->seed(DefaultMediaSizeSeeder::class);
        $this->asUser();

        $response = $this->getJson(route('content.media-sizes.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, count($response->json()));
    }
}
