<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Foundation\Testing\Concerns\InteractsWithTime;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Models\Media;
use Mtc\ContentManager\Models\MediaTag;
use Mtc\ContentManager\Models\MediaUse;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class MediaLibraryControllerTest extends TestCase
{
    use InteractsWithTime;
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();

        Media::factory(50)->create();
        $response = $this->get(route('content.media-library.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('files', $response->json());
        $this->assertArrayHasKey('tags', $response->json());
        $this->assertArrayHasKey('upload_dates', $response->json());
        $this->assertArrayHasKey('types', $response->json());
        $this->assertArrayHasKey('data', $response->json('files'));
        $this->assertArrayHasKey('total', $response->json('files'));
        $this->assertEquals(50, $response->json('files.total'));
    }

    public function testShow()
    {
        $this->asUser();

        Media::factory(50)->create();
        $response = $this->get(route('content.media-library.show'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('data', $response->json());
        $this->assertArrayHasKey('total', $response->json());
        $this->assertEquals(50, $response->json('total'));
    }

    public function testShowUses()
    {
        $this->asUser();

        $media = Media::factory()->create();
        MediaUse::factory(10)->create(['media_id' => $media->id]);
        $response = $this->get(route('content.media-library.show-uses', $media->id));

        $response->assertStatus(200);
        $this->assertArrayHasKey('data', $response->json());
        $this->assertCount(10, $response->json('data'));

    }

    public function testShowUploadDateFilter()
    {
        $this->asUser();

        Media::factory(5)->create();

        $this->travelTo('2020-11-01');
        Media::factory(3)->create();

        $this->travelTo('2022-01-01');
        Media::factory(4)->create();

        $response = $this->get(route('content.media-library.show', ['upload_date' => '2020-11']));
        $this->assertEquals(3, $response->json('total'));

        $response = $this->get(route('content.media-library.show', ['upload_date' => '2022-01']));
        $this->assertEquals(4, $response->json('total'));
    }

    public function testShowTagFilter()
    {
        $this->asUser();

        Media::factory(10)->create();
        $media1 = Media::factory()->create();
        $media2 = Media::factory()->create();
        $media3 = Media::factory()->create();
        MediaTag::factory()->create(['media_id' => $media1->id, 'tag' => 'foo']);
        MediaTag::factory()->create(['media_id' => $media2->id, 'tag' => 'foo']);
        MediaTag::factory()->create(['media_id' => $media3->id, 'tag' => 'foo']);
        $response = $this->get(route('content.media-library.show', ['tag' => 'foo']));

        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('total'));
    }

    public function testShowTypeFilter()
    {
        $this->asUser();

        Media::factory(5)->create();
        Media::factory(3)->create(['type' => 'image']);
        Media::factory(4)->create(['type' => 'document']);

        $response = $this->get(route('content.media-library.show', ['type' => 'image']));
        $this->assertEquals(3, $response->json('total'));

        $response = $this->get(route('content.media-library.show', ['type' => 'document']));
        $this->assertEquals(4, $response->json('total'));
    }

    public function testBulkDelete()
    {
        $this->asUser();
        $media_file = Media::factory()->create(['src' => 'avatar.jpg']);
        $media_file2 = Media::factory()->create(['src' => 'avatar2.jpg']);
        $media_file3 = Media::factory()->create(['src' => 'avatar3.jpg']);
        $response = $this->deleteJson(route('content.media-library.bulk-delete'), [
            'id' => [
                $media_file->id,
                $media_file2->id,
                $media_file3->id,
            ]
        ]);
        $response->assertStatus(200);
        $exists = Media::query()
            ->whereIn('id', [
                $media_file->id,
                $media_file2->id,
                $media_file3->id,
            ])
            ->exists();
        $this->assertFalse($exists);
    }

}
