<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\TemplateFactory;
use Mtc\ContentManager\Models\Comment;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\PageStatus;
use Mtc\ContentManager\Traits\ModelSortAndFilter;

abstract class Template extends Model
{
    use HasFactory;
    use ModelSortAndFilter;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'page_templates';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'name',
        'slug',
        'description',
        'seo_defaults',
        'meta',
        'updated_by',
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'seo_defaults' => 'array',
        'meta' => 'array',
        'updated_at' => 'datetime',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
    ];

    protected $appends = [
        'updated_at_formatted'
    ];

    /**
     * Model factory
     *
     * @return TemplateFactory
     */
    protected static function newFactory()
    {
        return TemplateFactory::new();
    }

    /**
     * Relationship with template content elements
     *
     * @return HasMany
     */
    public function elements(): HasMany
    {
        return $this->hasMany(Config::get('pages.template_element_model'), 'template_id');
    }

    /**
     * Relationship with content pages using this template
     *
     * @return HasMany
     */
    public function pages(): HasMany
    {
        return $this->hasMany(Config::get('pages.page_model'), 'template_id');
    }

    /**
     * Relationship with user who updated the template last
     *
     * @return BelongsTo
     */
    public function updatedBy(): BelongsTo
    {
        return $this->belongsTo(config('auth.providers.users.model'), 'updated_by');
    }

    /**
     * Relationship with Comments for page
     *
     * @return MorphMany
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Comment::class, 'commentable');
    }

    /**
     * Scope - WithActivePageCount()
     * Adds active_page_usage column with all active page count using this template
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithActivePageCount(Builder $query)
    {
        return $query->addSelect([
            'active_page_usage' => Page::query()
                ->selectRaw('COUNT(*)')
                ->where('status', PageStatus::PUBLISHED)
                ->whereColumn('template_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * Scope - WithInactivePageCount()
     * Adds inactive_page_usage column with all inactive page count using this template
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithInactivePageCount(Builder $query)
    {
        return $query->addSelect([
            'inactive_page_usage' => Page::query()
                ->selectRaw('COUNT(*)')
                ->where('status', '!=', PageStatus::PUBLISHED)
                ->whereColumn('template_id', $this->getTable() . '.id')
        ]);
    }

    /**
     * Scope - withCommentCount()
     * Adds comment_count column with all comments attached to template
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithCommentCount(Builder $query)
    {
        return $query->addSelect([
            'comment_count' => Comment::query()
                ->selectRaw('COUNT(*)')
                ->whereColumn('commentable_id', $this->getTable() . '.id')
                ->where('commentable_type', $this->getMorphClass())
        ]);
    }

    /**
     * updated_at_formatted
     * Displays updated_at in diff with now
     *
     * @return mixed
     */
    public function getUpdatedAtFormattedAttribute()
    {
        return $this->updated_at?->diffForHumans();
    }
}
