<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Mtc\ContentManager\Http\Requests\SaveImageSizesRequest;
use Mtc\ContentManager\Models\MediaSize;

class MediaSizeController
{
    use ValidatesRequests;

    /**
     * Show list of all supported media sizes
     *
     * @return \Illuminate\Database\Eloquent\Collection|MediaSize[]
     */
    public function index()
    {
        return MediaSize::all();
    }


    /**
     * Store sizes from array
     *
     * @param SaveImageSizesRequest $request
     * @return \Illuminate\Database\Eloquent\Collection|MediaSize[]
     */
    public function store(SaveImageSizesRequest $request)
    {
        collect($request->input('sizes'))
            ->each(function ($size) {
                $dimensions = [
                    'width' => $size['width'],
                    'height' => $size['height'],
                ];
                $additions = [
                    'label' => $size['label']
                ];
                MediaSize::query()->updateOrCreate($dimensions, $additions);
            });
        return MediaSize::all();
    }

    /**
     * Remove size
     *
     * @param MediaSize $media_size
     * @return \Illuminate\Database\Eloquent\Collection|MediaSize[]
     * @throws \Exception
     */
    public function destroy(MediaSize $media_size)
    {
        $media_size->delete();
        return MediaSize::all();
    }

    /**
     * Show sizes that exist for given dimensions
     *
     * @param Request $request
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection
     */
    public function forDimensions(Request $request)
    {
        $query = MediaSize::query();
        collect($request->input('sizes'))
            ->map(function ($size) {
                $dimensions = explode('x', $size);
                return [
                    'width' => $dimensions[0],
                    'height' => $dimensions[1],
                ];
            })
            ->each(function ($dimensions) use ($query) {
                $query->orWhere(function ($query) use ($dimensions) {
                    $query->where('width', $dimensions['width'])
                        ->where('height', $dimensions['height']);
                });
            });

        return $query->get();
    }
}
