<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Menu;
use Mtc\ContentManager\Models\MenuEntry;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class ElementControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();
        ContentElement::factory(5)->create([
            'drag_and_drop' => true,
        ]);
        $response = $this->getJson(route('content.elements.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, $response->json('total'));
        $this->assertArrayHasKey('data', $response->json());
    }

    public function testShow()
    {
        $this->asUser();
        $element = ContentElement::factory()->create([
            'title' => 'foo',
            'subtitle' => 'baz',
            'category' => 'bar',
            'is_enabled' => true,
        ]);
        $response = $this->getJson(route('content.elements.show', $element));

        $response->assertStatus(200);
        $this->assertEquals('foo', $response->json('data.title'));
        $this->assertEquals('baz', $response->json('data.subtitle'));
        $this->assertEquals('bar', $response->json('data.category'));
        $this->assertTrue($response->json('data.is_enabled'));
        $this->assertArrayHasKey('fields', $response->json('data'));
    }

    public function testStore()
    {
        $this->asUser();
        $response = $this->postJson(route('content.elements.store'), [
            'title' => 'foo',
        ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('data.title'));
        $this->assertEquals(1, ContentElement::query()->count());
    }

    public function testUpdate()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        $response = $this->putJson(route('content.elements.update', $element), [
            'title' => 'foo',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'name' => 'lorem',
                    'field_type' => 'ipsum'
                ]
            ]
        ]);

        $response->assertStatus(200);
        $element->refresh();
        $this->assertEquals('foo', $element->title);
        $this->assertEquals('baz', $element->subtitle);
        $this->assertEquals('bar', $element->category);
        $this->assertTrue($element->is_enabled);
        $this->assertEquals(1, $element->fields()->count());
    }

    public function testDestroy()
    {
        $this->asUser();
        $element = ContentElement::factory()->create();
        $response = $this->deleteJson(route('content.elements.destroy', $element));

        $response->assertStatus(200);
        $this->assertEquals('success', $response->json('status'));
        $this->assertFalse(ContentElement::query()->where('id', $element)->exists());
    }

    public function testCopy()
    {
        $this->asUser();
        $element = ContentElement::factory()->create();
        ContentElementField::factory(3)->create(['element_id' => $element->id]);
        $response = $this->postJson(route('content.elements.copy', $element), [
            'title' => 'foo'
        ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('data.title'));
        $this->assertEquals(2, ContentElement::query()->count());
        $this->assertEquals(3, ContentElementField::query()->where('element_id', $response->json('data.id'))->count());
    }

    public function testCheckUsage()
    {
        $this->asUser();
        $element = ContentElement::factory()->create();
        $response = $this->getJson(route('content.elements.show-usage', $element));

        $response->assertStatus(200);

        $this->assertEquals([
            'templates' => [],
            'freeform_pages' => [],
            'global_elements' => [],
            'menus' => [],
        ], $response->json());

        $menu = Menu::factory()->create();
        MenuEntry::factory()->create(['menu_id' => $menu->id, 'element_id' => $element->id]);
        GlobalContent::factory(3)->create(['element_id' => $element->id]);

        $response = $this->getJson(route('content.elements.show-usage', $element));
        $response->assertStatus(200);
        $this->assertEquals(1, count($response->json('menus')));
        $this->assertEquals(3, count($response->json('global_elements')));
    }

    public function testFullUsage()
    {
        $this->asUser();
        $element = ContentElement::factory()->create();
        GlobalContent::factory(30)->create(['element_id' => $element->id]);

        $response = $this->getJson(route('content.elements.show-usage', $element));
        $response->assertStatus(200);
        $this->assertEquals(10, count($response->json('global_elements')));

        $response = $this->getJson(route('content.elements.show-usage-for-type', [$element, 'global_elements']));
        $response->assertStatus(200);
        $this->assertEquals(30, count($response->json()));
    }

    public function testIcons()
    {
        $this->asUser();
        Storage::fake('content-icons');

        $response = $this->getJson(route('content.elements.icons'));
        $response->assertStatus(200);
        $this->assertIsArray($response->json());
        $this->assertEquals(0, count($response->json()));

        Storage::disk('content-icons')->put('foo.png', UploadedFile::fake()->image('foo.png'));
        Storage::disk('content-icons')->put('baz.png', UploadedFile::fake()->image('baz.png'));
        Storage::disk('content-icons')->put('bar.png', UploadedFile::fake()->image('bar.png'));

        $response = $this->getJson(route('content.elements.icons'));
        $response->assertStatus(200);
        $this->assertEquals(3, count($response->json()));
    }
}
