<?php

namespace Mtc\ContentManager;

use Exception;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Facades\Media;

class ImageSize
{
    /**
     * ImageSize constructor
     *
     * @param int $width
     * @param int|null $height
     * @param bool $allow_compression
     */
    public function __construct(
        protected int $width = 0,
        protected ?int $height = null,
        protected bool $allow_compression = true
    ) {
    }

    /**
     * Create instance from array
     *
     * @param array $config
     * @return static
     */
    public static function fromArray(array $config): self
    {
        $height = isset($config['height']) && $config['height'] !== null ? (int) $config['height'] : null;
        return new self($config['width'] ?? 0, $height, $config['allow_compression'] ?? true);
    }

    /**
     * Find destination for size on path
     *
     * @param string $fileName
     * @return string
     * @throws Exception
     */
    public function fileDestination(string $fileName, ?string $path = null, ?bool $legacyFormat = false): string
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->path($this->pathOnDisk($fileName, $path, $legacyFormat));
    }

    /**
     * Find the file url for size
     *
     * @param string $fileName
     * @return string
     */
    public function fileUrl(string $fileName, ?string $path = '', ?bool $legacyFormat = false): string
    {
        return Storage::disk(Config::get('filesystems.default_media'))
            ->url($this->pathOnDisk($fileName, $path, $legacyFormat));
    }

    /**
     * Find the path on disk for the file
     *
     * @param string $fileName
     * @return string
     */
    public function pathOnDisk(string $fileName = '', string $path = '', ?bool $legacyFormat = false): string
    {
        $dimensionString = $this->getDimensionString();
        return $legacyFormat
            ?  rtrim(Media::storagePrefix(), '/') . "/{$dimensionString}/$fileName"
            : rtrim(str_replace("/orig/", "/{$dimensionString}/", $path), '/') . "/$fileName";
    }

    /**
     * Get width of the ImageSize
     *
     * @return int
     */
    public function getWidth(): int
    {
        return $this->width;
    }

    /**
     * Get height of the ImageSize
     *
     * @return int|null
     */
    public function getHeight(): ?int
    {
        return $this->height;
    }

    /**
     * return whether size allows compression on image when resizing
     *
     * @return bool
     */
    public function getAllowCompression(): bool
    {
        return $this->allow_compression;
    }

    /**
     * Check if this is a width-only resize (no height specified)
     *
     * @return bool
     */
    public function isWidthOnly(): bool
    {
        return $this->height === null;
    }

    /**
     * Get the dimension string for path generation
     *
     * @return string
     */
    public function getDimensionString(): string
    {
        return $this->width . 'x' . ($this->height ?? 'Auto');
    }
}
