<?php

namespace Mtc\ContentManager;

use Illuminate\Foundation\Auth\User;
use Mtc\ContentManager\Notifications\NewCommentNotification;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\Comment;

class CommentRepository
{
    /**
     *
     * @param Comment $comment
     */
    public function __construct(protected Comment $commentModel)
    {
        //
    }

    /**
     * Add new comment
     *
     * @param string $text
     * @param int $author
     * @param Model $model
     * @param int|null $parentId
     * @return Model
     */
    public function add(string $text, int $author, Model $model, ?int $parentId = null, array $tagged_users = []): Model
    {
        /** @var Comment $comment */
        $comment = $this->commentModel
            ->newQuery()
            ->create([
                'commentable_id' => $model->id,
                'commentable_type' => $model->getMorphClass(),
                'parent_id' => $parentId,
                'author_id' => $author,
                'content' => $text
            ]);

        if (!empty($tagged_users)) {
            $comment->taggedUsers()->attach($tagged_users);
        } else {
            $comment->taggedUsers()->attach($this->getTaggedUsers($text));
        }
        $this->notifyAbout($comment, $author);

        return $comment;
    }

    /**
     * Get Comments for model
     *
     * @param string $commentable
     * @param int $id
     * @return Collection
     */
    public function get(string $commentable, int $id, bool $onlyActive = true): Collection
    {
        return $this->commentModel
            ->newQuery()
            ->whereNull('parent_id')
            ->whereNull('resolved_at')
            ->with([
                'author',
                $onlyActive ? 'activeChildren.author' : 'children.author',
                'taggedUsers',
                $onlyActive ? 'activeChildren.taggedUsers' : 'children.taggedUsers',
            ])
            ->where('commentable_id', $id)
            ->where('commentable_type', $commentable)
            ->oldest()
            ->get();
    }

    /**
     * Get Comments for model
     *
     * @param Model $model
     * @return Collection
     */
    public function getForModel(Model $model, bool $onlyActive = true): Collection
    {
        return $this->commentModel
            ->newQuery()
            ->whereNull('parent_id')
            ->whereNull('resolved_at')
            ->with([
                'author',
                $onlyActive ? 'activeChildren.author' : 'children.author',
                'taggedUsers',
                $onlyActive ? 'activeChildren.taggedUsers' : 'children.taggedUsers',
            ])
            ->where('commentable_id', $model->id)
            ->where('commentable_type', $model->getMorphClass())
            ->oldest()
            ->get();
    }

    /**
     * Mark a comment as resolved
     *
     * @param int $id
     * @return int
     */
    public function markResolved(int $id)
    {
        return $this->commentModel
            ->newQuery()
            ->where('id', $id)
            ->update([
                'resolved_at' => Carbon::now()
            ]);
    }

    /**
     * Notify relevant users about a comment
     *
     * @param Comment $comment
     */
    protected function notifyAbout(Comment $comment, int $currentUserId): void
    {
        if ($comment->parent_id > 0) {
            $recipients = $this->getThreadParticipants($comment);
        } else {
            $recipients = $comment->commentable->users ?? collect([]);
        }

        $recipients->merge($comment->taggedUsers)
            ->reject(fn(User $user) => $user->id === $currentUserId)
            ->each(function (User $user) use ($comment) {
                $user->notify(new NewCommentNotification($comment));
            });
    }

    /**
     * Get list of users participating in a comment thread
     *
     * @param Comment $comment
     * @return Collection
     */
    protected function getThreadParticipants(Comment $comment): Collection
    {
        $comment->parent->load('children.author');
        return $comment->parent->children->pluck('author')
            ->prepend($comment->parent->author)
            ->unique();
    }

    protected function getTaggedUsers(string $text)
    {
        return [];
    }
}
