<?php

namespace Mtc\ChequePayment;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

/**
 * Cheque / Money Order Payment Gateway
 *
 * @package  Mtc\ChequePayment
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class ChequePayment implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return App::make(config('cheque.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $invoice = template('cheque/payment_form.twig', [
            'invoice' => $invoice
        ]);
        return new PaymentForm($invoice, 'template', [
            'name' => __('cheque::cheque.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        return [
            'provider' => 'cheque',
            'amount' => $invoice->getOutstandingAmount(),
            'reference' => config('cheque.reference_prefix') . $invoice->getId(),
            'details' => [],
        ];

    }

}
