<?php

namespace Mtc\Checkout\Invoice;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Mtc\Checkout\Invoice;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;

/**
 * Class Item
 *
 * @package Mtc\Checkout\Invoice
 */
class Item extends Model
{
    use HasPrices;
    use SoftDeletes;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'invoice_items';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'purchasable_id',
        'purchasable_type',
        'parent_id',
        'parent_type',
        'original_value',
        'original_value_ex_vat',
        'value',
        'value_ex_vat',
        'quantity',
        'line_total',
        'line_total_ex_vat',
        'vat_rate',
        'details',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'details' => 'array'
    ];

    /**
     * Price fields
     *
     * @var array
     */
    protected $price_fields = [
        'value_ex_vat' => 'value',
        'original_value_ex_vat' => 'original_value',
        'line_total_ex_vat' => 'line_total',
    ];

    /**
     * column that sets vat rate on item
     *
     * @var string
     */
    protected $tax_attribute = 'vat_rate';

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::retrieved(function (self $order_item) {
            if ($order_item->invoice->use_ex_vat) {
                $order_item->load_ex_vat_price = true;
            }
        });
    }

    /**
     * Relationship with the invoice
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * Relationship with the purchased entity
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function purchasable()
    {
        return $this->morphTo();
    }

    /**
     * Relationship with the parent (owner) of this invoice item
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function parent()
    {
        return $this->morphTo();
    }

    /**
     * Get sku of the purchased entity
     *
     * @return string
     */
    public function getPurchasableSkuAttribute()
    {
        return $this->purchasable ? $this->purchasable->getSku() : '';
    }


    /**
     * Check how much the value has been discounted on invoice
     *
     * @return float|int
     */
    public function getDiscountPercentageAttribute()
    {
        return round(100 - (100 * $this->value / $this->original_value), 0);
    }

    /**
     * Get the item price after discounts
     *
     * @return float|mixed
     * @throws \Exception
     */
    public function getItemPriceAfterDiscountsAttribute()
    {
        // Currently supported only for shop orders
        if ($this->parent_type !== \Mtc\Orders\Item::class) {
            throw new \Exception('Only shop items supported for ->item_price_after_discounts');
        }

        $price =  $this->parent->paid_price->raw(true)
            - ($this->parent->amount_discount / $this->parent->quantity)
            - ($this->parent->paid_price->raw(true) * $this->parent->percentage_discount * 0.01);

        $calculated_price = new Price($price, $this->parent->paid_price->taxRate());
        $calculated_price->calculate();
        return $calculated_price;
    }

    /**
     * Get the line total after discounts
     *
     * @return float|mixed
     * @throws \Exception
     */
    public function getLinePriceAfterDiscountsAttribute()
    {
        // Currently supported only for shop orders
        if ($this->parent_type !== \Mtc\Orders\Item::class) {
            throw new \Exception('Only shop items supported for ->line_price_after_discounts');
        }

        $price =  $this->parent->paid_price->raw(true) * $this->quantity
            - ($this->parent->amount_discount / $this->parent->quantity * $this->quantity)
            - ($this->parent->paid_price->raw(true) * $this->quantity * $this->parent->percentage_discount * 0.01);

        $calculated_price = new Price($price, $this->parent->paid_price->taxRate());
        $calculated_price->calculate();
        return $calculated_price;
    }
}
