<?php

namespace Mtc\Checkout;

use Illuminate\Support\Manager;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;

/**
 * Class PaymentManager
 *
 * @package Mtc\Checkout
 */
class PaymentManager extends Manager
{
    /**
     * @var array
     */
    protected $applicable_drivers = [];

    /**
     * Get the default driver name.
     *
     * @return string
     */
    public function getDefaultDriver()
    {
        return $this->app['config']['checkout.gateway'];
    }

    /**
     * Filter out applicable gateways
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     */
    public function getApplicableDrivers(InvoiceRepositoryContract $invoice, $payable = null)
    {
        $this->createMissingDrivers();
        $this->applicable_drivers = collect($this->drivers)
            ->filter(function ($driver) use ($invoice, $payable) {
                return $driver instanceof PaymentGateway
                    && $driver->isApplicable($invoice, $payable);
            })
            ->toArray();

        return $this->applicable_drivers;
    }

    /**
     * Get the list of applicable payment forms for the payment
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return \Illuminate\Support\Collection
     */
    public function getApplicablePaymentForms(InvoiceRepositoryContract $invoice, PayableContract $payable = null)
    {
        return collect($this->getApplicableDrivers($invoice, $payable))
            ->map(function (PaymentGateway $driver) use ($invoice) {
                return $driver->getPaymentForm($invoice);
            });
    }

    /**
     * Build up drivers that are not yet created
     */
    protected function createMissingDrivers()
    {
        collect(array_diff(array_keys($this->customCreators), array_keys($this->drivers)))
            ->each(function ($driver) {
                $this->driver($driver);
            });
    }


    /**
     * Set the default session driver name.
     *
     * @param  string  $name
     * @return void
     */
    public function setActiveDriver($name)
    {
        $this->app['config']['checkout.gateway'] = $name;
    }
}
