<?php

namespace Mtc\Checkout\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\SplitInvoiceFactoryContract;
use Mtc\Checkout\Events\PaidInvoice;
use Mtc\Checkout\Invoice;
use Mtc\Checkout\InvoiceStatus;

/**
 * Class InvoiceController
 *
 * @package Mtc\Checkout
 */
class InvoiceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Model $invoice_model
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, Invoice $invoice_model)
    {
        /** @var Builder $query */
        $query = $invoice_model::query()
            ->search($request)
            ->latest();
        $this->page_meta['title'] = __('checkout::invoices.manage_invoices');
        return template('admin/invoices/index.twig', [
            'invoices' => $query->paginate(config('invoices.admin.pagination')),
            'page_meta' => $this->page_meta,
            'extra_actions' => collect(\event('invoices_index_extra_actions'))->filter(),
        ]);
    }

    /**
     * View Invoice
     *
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($invoice_id, InvoiceRepositoryContract $invoice)
    {
        $invoice->load($invoice_id);
        $payable = $invoice->getPayable();

        $this->page_meta['title'] = 'View Invoice ' . $invoice->getReference();
        return template('admin/invoices/show.twig', [
            'invoice' => $invoice->getModel(),
            'page_meta' => $this->page_meta,
            'link_to_payable' => $payable->getAdminLink(),
            'payable_name' => $payable->getName(),
        ]);
    }

    /**
     * Print an invoice
     *
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function print(Request $request, $invoice_id, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice = $invoice_repository->load($invoice_id);
        $invoice->template = $request->input('template', config('invoices.default_template'));
        $invoice->save();

        return template($invoice->template, [
            'invoice' => $invoice,
        ]);
    }

    /**
     * Print an invoice
     *
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function saveNote(Request $request, $invoice_id, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice = $invoice_repository->load($invoice_id);

        $details = $invoice->details;
        $details['note'] = $request->input('note');
        $invoice->details = $details;
        $invoice->save();

        session()->flash('success', __('checkout::invoices.note_saved'));
        return redirect()->back();
    }

    /**
     * Send an invite to pay
     *
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function invite(Request $request, $invoice_id, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice = $invoice_repository->load($invoice_id);
        if ($invoice_repository->inviteToPay()) {
            session()->flash('success', __('checkout::invoices.invite_email_sent'));
        }

        return redirect()->back();
    }

    /**
     * Void an invoice
     *
     * @param Request $request
     * @param Invoice $invoice
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Request $request, Invoice $invoice)
    {
        $invoice->status = InvoiceStatus::VOIDED;
        $invoice->save();

        session()->flash('success', __('checkout::invoices.invoice_voided'));
        return redirect()->back();
    }

    /**
     * Mark payment as confirmed
     *
     * Also reduce outstanding amount
     *
     * @param Invoice $invoice
     * @param Invoice\Payment $payment
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function confirmPayment(Invoice $invoice, Invoice\Payment $payment, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice_repository->load($invoice->id);
        $payment->confirmed_at = now();
        $payment->save();

        $invoice->outstanding_amount -= $payment->amount;
        if ($invoice->outstanding_amount <= 0) {
            $invoice->status = InvoiceStatus::PAID;

            Event::dispatch(new PaidInvoice($invoice_repository, $payment->reference, $payment->toArray()));
        }
        $invoice->save();

        session()->flash('success', __('checkout::invoices.payment_confirmed'));
        return redirect()->back();
    }

    /**
     * Create a manual payment for this invoice to mark it as paid
     *
     * @param Invoice $invoice
     * @param InvoiceRepositoryContract $invoice_repository
     * @return \Illuminate\Http\RedirectResponse
     */
    public function confirmManually(Invoice $invoice, InvoiceRepositoryContract $invoice_repository)
    {
        $invoice_repository->load($invoice->id);
        $payment = $invoice->payments()
            ->create([
                'provider' => 'manual-order',
                'amount' => $invoice_repository->getOutstandingAmount(),
                'reference' => config('orders.manual_order_reference_prefix') . $invoice_repository->getId(),
                'details' => [],
                'confirmed_at' => now(),
            ]);

        $invoice->outstanding_amount -= $payment->amount;
        if ($invoice->outstanding_amount <= 0) {
            $invoice->status = InvoiceStatus::PAID;

            Event::dispatch(new PaidInvoice($invoice_repository, $payment->reference, $payment->toArray()));
        }
        $invoice->save();

        session()->flash('success', __('checkout::invoices.payment_confirmed'));
        return redirect()->back();
    }

    /**
     * Display a listing of failed payments
     *
     * @param Model $invoice_model
     * @return \Illuminate\Http\Response
     */
    public function showFailed(Request $request)
    {
        $query = Invoice\Payment::query()
            ->latest('failed_at')
            ->with('invoice')
            ->whereNotNull('failed_at');

        $this->page_meta['title'] = __('checkout::invoices.failed_payments');
        return template('admin/invoices/failed_payments.twig', [
            'payments' => $query->paginate(config('invoices.admin.pagination')),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * View section for splitting invoice into multiple
     *
     * @param Invoice $invoice
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function splitView(Invoice $invoice)
    {
        $invoice->load([
            'items',
            'lines'
        ]);
        $this->page_meta['title'] = __('checkout::invoices.split_invoice');
        return template('admin/invoices/split_invoice.twig', [
            'invoice' => $invoice,
            'page_meta' => $this->page_meta,
        ]);
    }


    /**
     * View section for splitting invoice into multiple
     *
     * @param Invoice $invoice
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function split(Request $request, Invoice $invoice)
    {
        App::make(SplitInvoiceFactoryContract::class)->create($request, $invoice);

        session()->flash('success', __('checkout::invoices.split_success_message'));
        return redirect()->to(route('invoices.show', [ $invoice->id ]));
    }

}

