<?php

namespace Mtc\BitPay;

use BitPaySDK\Client;
use BitPaySDK\Model\Invoice\Buyer;
use BitPaySDK\Model\Invoice\Invoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\File;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;

class BitPay implements PaymentGateway
{
    public const CONFIG_FILE_NAME = 'bitpay-config.json';


    /**
     * @param InvoiceRepositoryContract $invoice
     * @param \Mtc\Checkout\Contracts\PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= config('bitpay.minimal_transaction_amount')) {
            return false;
        }

        if ($this->configFile() === false) {
            return false;
        }

        return App::make(config('bitpay.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     * @throws \BitPaySDK\Exceptions\BitPayException
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        return new PaymentForm('bitpay-payment', 'vue-component', [
            'invoice_id' => $invoice->getId(),
            'name' => __('bitpay::bitpay.payment_option_name'),
            'test_mode' => true,
            'bitpay_url' => $this->createBitPayInvoice($invoice)
        ]);
    }

    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $invoice_id = $invoice->getModel()->details['bitpay_invoice_id'] ?? null;

        if (empty($invoice_id)) {
            return [];
        }

        return $this->getInvoiceDetails($invoice_id, $invoice);
    }

    /**
     * Create invoice on BitPay
     *
     * @param InvoiceRepositoryContract $invoice
     * @return mixed
     * @throws \BitPaySDK\Exceptions\BitPayException
     */
    protected function createBitPayInvoice(InvoiceRepositoryContract $invoice)
    {
        $bitpay = Client::create()->withFile($this->configFile());
        $bitpay_invoice = new Invoice($invoice->getOutstandingAmount(), $invoice->getCurrency());
        $bitpay_invoice->setOrderId($invoice->getReference());

        $redirect_url = route('charge_payment', [
            'invoice_id' => $invoice->getId(),
            'gateway' => 'bitpay',
        ]);

        $bitpay_invoice->setRedirectURL($redirect_url);
        $bitpay_invoice->setNotificationURL(route('bitpay_notification', ['invoice' => $invoice->getId()]));

        $buyer = new Buyer();
        $buyer->setEmail($invoice->getEmail());

        $bitpay_invoice->setBuyer($buyer);
        $bitpay_invoice = $bitpay->createInvoice($bitpay_invoice);

        $details = $invoice->getModel()->details ?? [];
        $details['bitpay_invoice_id'] = $bitpay_invoice->getId();
        $invoice->getModel()
            ->update([
                'details' => $details,
            ]);

        return $bitpay_invoice->getUrl();
    }

    /**
     * Check invoice details on BitPay to confirm it has been paid
     *
     * @param string $invoice_id
     * @param InvoiceRepositoryContract $invoice_repository
     * @return array
     * @throws \BitPaySDK\Exceptions\BitPayException
     */
    protected function getInvoiceDetails(string $invoice_id, InvoiceRepositoryContract $invoice_repository): array
    {
        $bitpay = Client::create()->withFile($this->configFile());
        $invoice = $bitpay->getInvoice($invoice_id);

        if ($invoice && in_array($invoice->getStatus(), config('bitpay.acceptable_payment_statuses', []))) {
            return [
                'provider' => 'bitpay',
                'amount' => $invoice_repository->getOutstandingAmount(),
                'currency_code' => $invoice_repository->getCurrency(),
                'amount_in_currency' => $invoice_repository->getOutstandingAmountInCurrency(),
                'reference' => $invoice->getId(),
                'confirmation_status' => $invoice->getStatus() === 'complete'
                    ? config('bitpay.transaction_finished_order_status')
                    : config('bitpay.transaction_received_order_status'),
            ];
        }

        return [];
    }

    /**
     * Path to BitPay config file
     *
     * @return false|string
     */
    protected function configFile()
    {
        $config_file = storage_path(self::CONFIG_FILE_NAME);

        return File::exists($config_file)
            ? $config_file
            : false;
    }
}
