<?php

namespace Mtc\BitPay\Console\Commands;

use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Mtc\BitPay\BitPay;
use BitPayKeyUtils\KeyHelper\PrivateKey;
use BitPayKeyUtils\Storage\EncryptedFilesystemStorage;
use Symfony\Component\Yaml\Yaml;

class CreateBitPayConfig extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'bitpay:create-config {--test}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create BitPay Config File';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle(): int
    {
        $this->saveConfig($this->generateConfig());
        return 0;
    }

    /**
     * @param array $data
     */
    protected function saveConfig(array $data)
    {
        File::put(storage_path(BitPay::CONFIG_FILE_NAME), json_encode($data));
    }

    /**
     * Generate new private key.
     * Make sure you provide an easy recognizable name to your private key
     * NOTE: In case you are providing the BitPay services to your clients,
     *       you MUST generate a different key per each of your clients
     *
     * WARNING: It is EXTREMELY IMPORTANT to place this key files in a very SECURE location
     **/
    protected function generateConfig(): array
    {
        $key_name = storage_path('bitpay-private-key.key');
        $private_key = new PrivateKey($key_name);
        $storageEngine = new EncryptedFilesystemStorage(config('bitpay.master_password'));

        try {
            //  Use the EncryptedFilesystemStorage to load the Merchant's encrypted private key with the Master Password
            $private_key = $storageEngine->load($key_name);
        } catch (Exception $ex) {
            //  Check if the loaded keys is a valid key
            if (!$private_key->isValid()) {
                $private_key->generate();
            }

            //  Encrypt and store it securely.
            //  This Master password could be one for all keys or a different one for each Private Key
            $storageEngine->persist($private_key);
        }

        /**
         * Generate the public key from the private key every time (no need to store the public key).
         **/
        try {
            $public_key = $private_key->getPublicKey();
        } catch (Exception $ex) {
            echo $ex->getMessage();
        }

        /**
         * Derive the SIN from the public key.
         **/
        try {
            $sin = $public_key->getSin()->__toString();
        } catch (Exception $ex) {
            echo $ex->getMessage();
        }

        /**
         * Use the SIN to request a pairing code and token.
         * The pairing code has to be approved in the BitPay Dashboard
         * THIS is just a cUrl example, which explains how to use the key pair for signing requests
         **/
        $base_url = $this->option('test') ? 'https://test.bitpay.com' : 'https://bitpay.com';
        $env = $this->option('test') ? 'Test' : 'Prod';

        $token = null;

        /**
         * Request a token for the Merchant facade
         */

        try {
            $postData = json_encode([
                'id' => $sin,
                'facade' => 'merchant',
            ]);

            $curl = curl_init($base_url . "/tokens");

            curl_setopt(
                $curl,
                CURLOPT_HTTPHEADER,
                [
                    'x-accept-version: 2.0.0',
                    'Content-Type: application/json',
                    'x-identity' => $public_key->__toString(),
                    'x-signature' => $private_key->sign($base_url . "/tokens" . $postData),
                ]
            );

            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'POST');
            curl_setopt($curl, CURLOPT_POSTFIELDS, stripslashes($postData));
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

            $result = curl_exec($curl);
            $result_data = json_decode($result, true);
            curl_close($curl);

            if (array_key_exists('error', $result_data)) {
                $this->error($result_data['error']);
                return [];
            }

            /**
             * Example of a pairing Code returned from the BitPay API
             * which needs to be APPROVED on the BitPay Dashboard before being able to use it.
             **/
            $token = $result_data['data'][0]['token'];
            $this->info('Merchant Facade');
            $this->info('  -> Pairing Code: ' . $result_data['data'][0]['pairingCode']);
            $this->info('  -> Token: ' . $token);
            /** End of request **/
        } catch (Exception $ex) {
            echo $ex->getMessage();
        }

        $this->info("Please, copy the above pairing code/s and approve on your BitPay Account at the following link:");
            $this->info($base_url . "/dashboard/merchant/api-tokens");
        $this->info("Once you have this Pairing Code/s approved you can move the'
            .' generated files to a secure location and start using the Client.");

        return [
            "BitPayConfiguration" => [
                "Environment" => $env,
                "EnvConfig" => [
                    'Prod' => [
                        "PrivateKeyPath" => $this->option('test') ? null : $key_name,
                        "PrivateKeySecret" => $this->option('test') ? null : config('bitpay.master_password'),
                        "ApiTokens" => [
                            "merchant" => $this->option('test') ? null : $token,
                            "payout" => null,
                        ],
                        "proxy" => null,
                    ],
                    'Test' => [
                        "PrivateKeyPath" => $this->option('test') ? $key_name : null,
                        "PrivateKeySecret" => $this->option('test') ? config('bitpay.master_password') : null,
                        "ApiTokens" => [
                            "merchant" => $this->option('test') ? $token : null,
                            "payout" => null,
                        ],
                        "proxy" => null,
                    ],
                ],
            ],
        ];
    }
}
