<?php

namespace Mtc\Basket;

use Mtc\Money\Price;

/**
 * Trait BasketItemPricing
 *
 * @package Mtc\Basket
 */
trait BasketItemPricing
{
    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getOriginalUnitPriceAttribute($with_tax = false)
    {
        return $this->purchasable->getPrice()->raw($with_tax);
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @param bool|null $with_tax
     * @return mixed
     */
    public function getUnitPriceAttribute($with_tax = null)
    {
        if (!empty($this->attributes['custom_price'])) {
            $with_tax = $with_tax ?? config('basket.custom_price_display_with_vat');
            return $this->custom_price->raw($with_tax);
        }
        return $this->purchasable->getPrice()->raw($with_tax);
    }

    /**
     * Get Unit price ex.VAT from purchasable object
     *
     * @return float
     */
    public function getUnitPriceExVatAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            return $this->custom_price->raw(false);
        }
        return round($this->purchasable->getPrice()->withoutTax(), 2);
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getOriginalPriceAttribute($with_tax = false)
    {
        return $this->purchasable->getOriginalPrice()->raw($with_tax);
    }

    /**
     * Get Original price ex.VAT from purchasable object
     *
     * @return float
     */
    public function getOriginalPriceExVatAttribute()
    {
        return round($this->purchasable->getOriginalPrice()->withoutTax(), 2);
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getLineTaxAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            if (!config('basket.custom_price_display_with_vat')) {
                $custom_price_config['price_entered_with_tax'] = false;
                $custom_price = new Price($this->attributes['custom_price'], null, $custom_price_config);
                $custom_price->calculate();
                return round($custom_price->tax(3), 2) * $this->quantity;
            }

            return $this->custom_price->tax(3) * $this->quantity;
        }
        return $this->purchasable->getPrice()->tax(3) * $this->quantity;
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getLineTotalExVatAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            return $this->custom_price->raw(false) * $this->quantity;
        }
        return $this->purchasable->getPrice()->raw(false) * $this->quantity;
    }

    /**
     * @return float
     */
    public function getLineTotalIncVatAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            return $this->custom_price->raw(true) * $this->quantity;
        }
        return $this->purchasable->getPrice()->raw(true) * $this->quantity;
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getLineTotalAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            return $this->custom_price->raw(config('basket.custom_price_display_with_vat')) * $this->quantity;
        }
        return $this->purchasable->getPrice()->raw(config('basket.subtotal_display_with_vat')) * $this->quantity;
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getLineOriginalAttribute()
    {
        return $this->purchasable->getOriginalPrice(config('basket.subtotal_display_with_vat'))->raw()
            * $this->quantity;
    }

    /**
     * Get Line Original ex.VAT from purchasable object
     *
     * @return float
     */
    public function getLineOriginalExVatAttribute()
    {
        return round($this->purchasable->getOriginalPrice()->withoutTax(), 2) * $this->quantity;
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getAvailableLineTotalAttribute()
    {
        if (!empty($this->attributes['custom_price'])) {
            return $this->custom_price->raw(config('basket.custom_price_display_with_vat'))
                * $this->available_quantity;
        }
        return $this->purchasable->getPrice()->raw(config('basket.subtotal_display_with_vat'))
            * $this->available_quantity;
    }

    /**
     * Get Discounted Values
     *
     * @return mixed
     */
    public function getDiscountedValues()
    {
        if (config('orders.use_discounts_per_item_line')) {
            $discounted_values = $this->basket->discounts
                ->filter(function ($basket_discount) {
                    return in_array($this->purchasable->node->id, $basket_discount->applicable_items);
                })
                ->map(function ($basket_discount) {
                    return $basket_discount->discount->getDiscountedValues($this);
                });

            return [
                'amount_discount' => $discounted_values->sum('discounted_amount'),
                'percentage_discount' => $discounted_values->sum('discounted_percent')
            ];
        }

        return null;
    }
}
