<?php

namespace Mtc\Basket;

use Illuminate\Database\Eloquent\Model;
use Mtc\Basket\Events\BasketItemCreating;
use Mtc\Basket\Events\BasketItemSaving;
use Mtc\Money\HasPrices;

/**
 * Class Item
 *
 * @package Mtc\Basket
 */
class Item extends Model
{
    use HasPrices;
    use BasketItemPricing;
    use BackOrderPricing;
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'basket_items';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * Touch (update) relationship when changes are made
     *
     * @var array
     */
    protected $touches = [
        'basket',
    ];

    /**
     * Cast attributes to types
     *
     * @var array
     */
    protected $casts = [
        'attribute_fields' => 'array',
        'data' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Append attributes to JSON
     *
     * @var array
     */
    protected $appends = [
        'url',
        'image',
        'name',
        'original_unit_price',
        'unit_price',
        'unit_price_ex_vat',
        'original_price',
        'original_price_ex_vat',
        'line_total',
        'available_line_total',
        'line_original',
        'line_original_ex_vat',
        'stock',
        'available_quantity',
        'backorder_quantity',
        'backorder_line_original',
        'backorder_line_total',
    ];

    /**
     * Hide attributes from ajax response
     *
     * @var array
     */
    protected $hidden = [
        'basket_id',
        'purchasable_id',
        'created_at',
        'updated_at',
        'purchasable',
    ];

    /**
     * Price auto-loading
     *
     * @var array
     */
    protected $price_fields = [
        'custom_price',
    ];

    /**
     * Dispatch event on specific actions
     *
     * @var array
     */
    protected $dispatchesEvents = [
        'creating' => BasketItemCreating::class,
        'saving' => BasketItemSaving::class,
    ];

    /**
     * Relationship with Basket
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basket()
    {
        return $this->belongsTo(config('basket.model', Basket::class));
    }

    /**
     * Relationship with item that is getting purchased
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function purchasable()
    {
        return $this->morphTo();
    }

    /**
     * Basket Item may have sub-lines to break down details
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lines()
    {
        return $this->hasMany(Line::class, 'basket_item_id');
    }

    /**
     * Set the quantity for the basket item
     *
     * @param $request_quantity
     * @param $append
     * @param $max_qty
     */
    public function setQuantity($request_quantity, $append, $max_qty)
    {
        $this->quantity = $append ? $this->quantity + $request_quantity : $request_quantity;

        if (config('basket.force_max_qty_when_adding_item_in_basket') && $this->quantity > $max_qty) {
            $this->quantity = $max_qty;
        }

        if ($this->quantity > config('basket.max_qty_of_item_in_basket')) {
            $this->quantity = config('basket.max_qty_of_item_in_basket');
        }
    }

    /**
     * Get Url attribute from purchasable attribute
     *
     * @return mixed
     */
    public function getUrlAttribute()
    {
        return $this->purchasable->getUrl();
    }

    /**
     * Get Url attribute from purchasable attribute
     *
     * @return mixed
     */
    public function getAvailableQuantityAttribute()
    {
        return ($this->stock < $this->quantity && config('basket.allow_backorders') === true) ? $this->stock : $this->quantity;
    }
    /**
     * Get Url attribute from purchasable attribute
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        return $this->purchasable->getName();
    }

    /**
     * Get Basket image from purchasable object
     *
     * @return mixed
     */
    public function getImageAttribute()
    {
        return $this->purchasable->getBasketImage();
    }

    /**
     * Get Tax rate name
     *
     * @return mixed
     */
    public function getTaxRateAttribute()
    {
        return $this->purchasable->getPrice()->taxRate();
    }

    /**
     * Get Price attribute from purchasable object
     *
     * @return mixed
     */
    public function getCanEditQuantityAttribute()
    {
        return true;
    }

    /**
     * Get stock of the purchasable
     *
     * @return mixed
     */
    public function getStockAttribute()
    {
        $stock = $this->purchasable->getStock();
        return $stock > 0 ? $stock : 0;
    }
}
