<?php

namespace Mtc\Basket\Repositories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Address;
use Mtc\Basket\Basket;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Contracts\Purchasable;
use Mtc\Basket\Item;
use Mtc\Core\Services\LocationService;

/**
 * Class BasketRepository
 *
 * @package Mtc\Basket\Repositories
 */
class BasketRepository implements BasketRepositoryInterface
{
    /**
     * List of validation errors
     *
     * @var array
     */
    protected $errors = [];

    /**
     * @var Basket
     */
    protected $basket;

    /**
     * BasketRepository constructor.
     */
    public function __construct()
    {
        $this->basket = App::make('basket');
    }

    /**
     * Fetch the current basket model
     *
     * @return Model
     */
    public function getModel(): Model
    {
        return $this->basket;
    }

    /**
     * Fetch the current basket model
     *
     * @return int
     */
    public function getId(): int
    {
        return $this->basket->id ?? 0;
    }

    /**
     * Get the member associated to the basket
     *
     * @return int
     */
    public function getMember()
    {
        return $this->basket->member;
    }

    /**
     * Find a basket by ID
     *
     * @param $basket_id
     */
    public function find($basket_id)
    {
        $class = App::make(config('basket.model'));
        $this->basket = $class->newQuery()->find($basket_id);
    }

    /**
     * Set generic info on Basket (e.g. email/phone)
     *
     * @param $details
     */
    public function setDetails($details)
    {
        $this->basket->fill($details);
        $this->basket->save();
    }

    /**
     * Set address info on Basket
     *
     * @param $type
     * @param $address_data
     */
    public function setAddress($type, $address_data)
    {
        $this->basket->addresses()
            ->updateOrCreate([
                'type' => $type
            ], $address_data);

        if ($type === Address::TYPE_SHIPPING) {
            $has_billing_address = $this->basket
                ->addresses()
                ->where('type', Address::TYPE_BILLING)
                ->exists();

            if (!$has_billing_address) {
                $address_data['type'] = Address::TYPE_BILLING;

                $this->basket->addresses()
                    ->updateOrCreate([
                        'type' => Address::TYPE_BILLING
                    ], $address_data);
            }
        }
    }

    /**
     * Check if basket has items
     *
     * @return bool
     */
    public function hasItems(): bool
    {
        return $this->basket->items()->count() > 0;
    }

    /**
     * Add an item to the basket. If basket doesn't exist, it will create it.
     *
     * @param Purchasable $purchasable item that implements Purchasable contract
     * @param integer $quantity Number of items to add to basket
     * @param bool $append Flag whether the quantities of the basket items should be appended
     * @param bool $force_new Flag to create the basket item as a new line
     *
     * @return void
     */
    public function addItem(Purchasable $purchasable, int $quantity, $append = true, $force_new = false)
    {
        // If we don't have a basket yet, spool it up.
        if (!$this->basket->exists) {
            $this->basket->save();

            if (config('shipping_manager.preemptive_shipping_calculation') ) {
                // Reload shipping address as it might be set via create event (e.g. member)
                $this->basket->load('shippingAddress');
                if (!$this->basket->shippingAddress) {
                    $detected_location = App::make(LocationService::class)->detectCountry();;
                    $address = config('basket.preemptive_shipping_address', []);
                    if (!empty($detected_location['code'])) {
                        $address['country'] = $detected_location['code'];
                    }
                    $this->basket->shippingAddress()->create($address);
                }
            }
        }

        /** @var Item $basket_item */
        $basket_item = $this->getAddBasketItemObject($purchasable, $force_new);
        $basket_item->attribute_fields = $purchasable->only($purchasable->getAttributeFields());
        $basket_item->sku = $purchasable->getSku();
        $basket_item->setQuantity($quantity, $append, $purchasable->getMaxQuantity());
        $basket_item->save();

        $this->removeDeliveryIfNotNeeded();
    }

    /**
     * @param $purchasable
     * @param $force_new
     * @return Model|\Illuminate\Database\Eloquent\Relations\HasMany
     */
    protected function getAddBasketItemObject($purchasable, $force_new)
    {
        $purchasable_type = array_search(get_class($purchasable), Relation::morphMap()) ?: get_class($purchasable);
        if ($force_new) {
            return $this->basket->items()->newModelInstance([
                'basket_id' => $this->basket->id,
                'purchasable_id' => $purchasable->id,
                'purchasable_type' => $purchasable_type,
            ]);
        }

        return $this->basket->items()
            ->firstOrNew([
                'purchasable_id' => $purchasable->id,
                'purchasable_type' => $purchasable_type,
            ]);
    }

    /**
     * Check if this item is in basket
     *
     * @param Purchasable $purchasable item that implements Purchasable contract
     * @param integer $quantity Number of items to add to basket
     * @return bool
     */
    public function hasItem(Purchasable $purchasable, $quantity = false)
    {
        // Check only class matching
        return $this->basket
                ->items
                ->filter(function ($item) use ($purchasable, $quantity) {
                    $quantity_match = true;
                    if ($quantity > 0) {
                        $quantity_match = $item->quantity == $quantity;
                    }

                    return $quantity_match
                        && get_class($item->purchasable) == get_class($purchasable)
                        && $item->purchasable->id == $purchasable->id;
                })
                ->count() > 0;

    }

    /**
     * Get the items in basket
     *
     * @return array
     */
    public function getItems()
    {
        return $this->basket->items;
    }

    /**
     * Remove all items from basket
     *
     * @return array
     */
    public function removeAllItems()
    {
        return $this->basket->items()->delete();
    }

    /**
     * Check if basket needs delivery
     *
     * @return bool
     */
    public function needsDelivery()
    {
        return $this->basket->getNeedsDeliveryAttribute();
    }

    /**
     * Remove delivery surcharge if it is not needed on basket
     */
    protected function removeDeliveryIfNotNeeded()
    {
        if ($this->needsDelivery() == false && $this->hasSurcharge(0, 'delivery')) {
            $this->removeSurchargeByType('delivery');
        }
    }

    /**
     * Change the quantities of basket items based on id => stock array
     *
     * @param array $quantity_array
     * @return void
     */
    public function updateItemQuantities($quantity_array)
    {
        collect($quantity_array)
            ->each(function ($quantity, $basket_item_id) {
                $item = $this->basket->items()->where('id', $basket_item_id);
                if ($quantity == 0) {
                    $item->delete();
                } else {
                    $item->update([
                        'quantity' => $quantity
                    ]);
                }

            });

        $this->removeDeliveryIfNotNeeded();

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Add discount to current Basket
     *
     * @param int $discount_id
     * @param string $discount_class
     * @param array|null $applicable_items
     * @return void
     */
    public function addDiscount(int $discount_id, string $discount_class, array $applicable_items = null)
    {
        $this->basket->discounts()
            ->updateOrCreate([
                'discount_id' => $discount_id,
                'discount_type' => $discount_class,
                'applicable_items' => $applicable_items
            ]);

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Check if Basket has discount
     *
     * @param int $discount_id
     * @param string $discount_class
     * @return bool
     */
    public function hasDiscount(int $discount_id, string $discount_class): bool
    {
        if ($discount_id == 0) {
            return $this->basket->discounts
                    ->where('discount_type', $discount_class)
                    ->count() > 0;
        }

        return $this->basket->discounts
                ->where('discount_id', $discount_id)
                ->where('discount_type', $discount_class)
                ->count() > 0;
    }

    /**
     * Remove discount from current basket
     *
     * @param int $discount_id
     * @param string $discount_class
     * @return void
     */
    public function removeDiscount(int $discount_id)
    {
        $this->basket->discounts
            ->find($discount_id)
            ->delete();

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Add surcharge to current basket
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return void
     */
    public function addSurcharge(int $surcharge_id, string $surcharge_class)
    {
        $this->basket->surcharges()
            ->updateOrCreate([
                'surcharge_id' => $surcharge_id,
                'surcharge_type' => $surcharge_class,
            ]);

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Check if basket has surcharge
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return bool
     */
    public function hasSurcharge(int $surcharge_id, string $surcharge_class): bool
    {
        if ($surcharge_id == 0) {
            return $this->basket->surcharges
                    ->where('surcharge_type', $surcharge_class)
                    ->count() > 0;
        }

        return $this->basket->surcharges
                ->where('surcharge_id', $surcharge_id)
                ->where('surcharge_type', $surcharge_class)
                ->count() > 0;
    }

    /**
     * Remove surcharge from current basket
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return void
     */
    public function removeSurcharge(int $surcharge_id)
    {
        $this->basket->surcharges()
            ->where('id', $surcharge_id)
            ->delete();

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Update surcharge for surcharge type
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return void
     */
    public function updateSurchargeByType($surcharge_type, $surcharge_id)
    {
        $this->basket->surcharges()
            ->updateOrCreate([
                'surcharge_type' => $surcharge_type
            ], [
                'surcharge_type' => $surcharge_type,
                'surcharge_id' => $surcharge_id,
            ]);

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Get the surcharge ID from type
     *
     * @param string $surcharge_type
     * @return int|null
     */
    public function getSurchargeIdFromType($surcharge_type)
    {
        return $this->basket
                ->surcharges
                ->where('surcharge_type', $surcharge_type)
                ->first()
                ->surcharge_id ?? null;
    }

    /**
     * Remove surcharge from current basket
     *
     * @param int $surcharge_id
     * @param string $surcharge_class
     * @return void
     */
    public function removeSurchargeByType($surcharge_type)
    {
        $this->basket->surcharges()
            ->where('surcharge_type', $surcharge_type)
            ->delete();

        // Refresh basket to ensure discounts/surcharges/totals update
        $this->basket->refresh();
    }

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getCostSubtotalAttribute()
    {
        return $this->basket->getCostSubtotalAttribute();
    }

    /**
     * Get the subtotal cost for all items in the basket.
     *
     * @return float
     */
    public function getCostTotalAttribute()
    {
        return $this->basket->getCostTotalAttribute();
    }

    /**
     * Get the basket weight.
     *
     * @return float
     */
    public function getWeight()
    {
        return $this->basket->items
            ->map(function (Item $item) {
                return $item->purchasable->getWeight() * $item->quantity;
            })
            ->sum();
    }

    /**
     * Get the destination address of the basket
     *
     * @return mixed
     */
    public function getDestinationAddress()
    {
        return $this->basket->shippingAddress;
    }

    /**
     * Attach additional data block to basket
     *
     * @param $data
     * @param string $key
     * @return mixed|void
     */
    public function setExtra($data, $key)
    {
        $this->basket->extra[$key] = $data;
    }

    /**
     * Get the customer name for this basket
     *
     * @return string
     */
    public function getCustomerName(): string
    {
        if ($this->basket->billingAddress) {
            return $this->basket->billingAddress->first_name . ' ' . $this->basket->billingAddress->last_name;
        }

        if ($this->basket->shippingAddress) {
            return $this->basket->shippingAddress->first_name . ' ' . $this->basket->shippingAddress->last_name;
        }

        return '';
    }

    /**
     * Get the vat rate applied to basket
     *
     * @return string
     */
    public function getVatValue(): string
    {
        return $this->basket->tax;
    }

    /**
     * Get the basket type
     *
     * @return mixed
     */
    public function getType()
    {
        return null;
    }


    /**
     * Set the member ID on basket
     *
     * @param $member_id
     * @return void
     */
    public function setMember($member_id)
    {
        $this->basket->member_id = $member_id;
        $this->basket->save();
    }

    /**
     * Add external gateway to basket
     *
     * @param $name
     * @param $context
     * @return void
     */
    public function setExternalGateway($name, $context)
    {
        $this->basket->external_gateways[$name] = $context;
    }

    /**
     * @return string
     */
    public function getCurrency(): string
    {
        return config('currencies.default_currency', '');
    }

    /**
     * @return mixed|void
     */
    public function allowOnlyAvailableItems()
    {
        $this->basket->load([
            'items.purchasable',
        ]);

        $changed_items = $this->basket->items
            ->map(function ($item) {
                if ($item->purchasable == null || $item->purchasable->isAvailable() == false) {
                    $item->delete();
                    return true;
                }
                return null;
            })
            ->filter();

        $this->basket->refresh();
    }

    /**
     * Set custom basket price for item - e.g. fixed price in order creation
     *
     * @param int $item_id
     * @param float $item_price
     */
    public function setCustomPrice($item_id, $item_price)
    {
        $this->basket->items()
            ->where('id', $item_id)
            ->update([
                'custom_price' => $item_price
            ]);

        $this->basket->refresh();
    }

    /**
     * Whether Ex vat costs should be used.
     * Necessary for order generation process
     *
     * @return bool
     */
    public function shouldUseExVat(): bool
    {
        return false;
    }

    /**
     * Set errors on basket
     *
     * @param array $errors
     */
    public function setErrors($errors = [])
    {
        $this->errors = array_merge($this->errors, $errors);
    }

    /**
     * Get validation errors
     *
     * @return array
     */
    public function getErrors()
    {
        return $this->errors;
    }
}
