<?php

namespace Mtc\Basket\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Facades\Basket;
use Mtc\Shop\Events\BeforeAddToBasketEvent;
use Mtc\Shop\Item\Size;

/**
 * Class LegacyBasketController
 *
 * Add plug-and-play functionality for some legacy basket actions
 *
 * @package Mtc\Basket\Http\Controllers
 */
class LegacyBasketController extends Controller
{
    /**
     * @param Request $request
     * @return array|\Illuminate\Http\RedirectResponse
     */
    public function addShopItem(Request $request)
    {
        $this->validate($request, [
            'id' => 'required|numeric|exists:items,id',
            'quantity' => 'numeric',
            'size.*' => 'numeric',
        ]);

        $item = new \Item();
        $item->Get_Item($request->input('id'));

        if ($this->checkForSizeError($request, $item)) {
            return $this->sizeErrorResponse($request, $item);
        }

        return $this->addItemInBasket($request, $item);
    }

    /**
     * Check if size info is ok
     *
     * @param Request $request
     * @param \Item $item
     * @return bool
     */
    protected function checkForSizeError(Request $request, \Item $item)
    {
        if (!item_has_sizes($item->id)) {
            return false;
        }

        // Check that array has size as an array with non-zero value
        return !$request->has('size') || (is_array($request->input('size')) && !array_search(true, $request->input('size')));

    }

    /**
     * Generate the size error response
     *
     * @param Request $request
     * @param \Item $item
     * @return array|\Illuminate\Http\RedirectResponse
     */
    protected function sizeErrorResponse(Request $request, \Item $item)
    {
        if ($request->wantsJson()) {
            $quantity_error = defined('QUICK_ADDTOBASKET') && QUICK_ADDTOBASKET === true && is_array($request->input('size'));
            return [
                'status' => 'failed',
                'error' => $quantity_error ? 'quantity' : 'size',
                'url' => ''
            ];
        }

        $params = [
            'error' => 'size'
        ];
        if ($request->has('color')) {
            $params['color'] = clean_page($request->input('color'));
        }
        if ($request->has('size')) {
            $params['size'] = clean_page($request->input('size'));
        }
        $url = $item->url . '?' . http_build_query($params);

        return redirect()->to($url);
    }

    /**
     * Add item in basket
     *
     * @param Request $request
     * @param \Item $item
     * @return array|\Illuminate\Http\RedirectResponse|mixed
     */
    protected function addItemInBasket(Request $request, \Item $item)
    {
        $params = [
            'id' => $item->id,
            'quantity' => 1,
            'PLU' => $item->epos_code
        ];

        if (is_array($request->input('size'))) {
            $sizes = collect($request->input('size'));
        } else {
            $sizes = [
                $request->input('size') => $request->input('quantity')
            ];
        }

        $this->putSizesInBasket($item, $sizes);

        if ($request->wantsJson()) {
            $response = [
                'status' => 'ok',
                'url' => ''
            ];
        } elseif (DIRECT_TO_BASKET === true) {
            $url = '/basket';
        } else {
            $_SESSION['added'] = 'added';
            $url = $item->url;
        }

        if ($request->has('quick_buy')) {
            return $request->wantsJson() == false ? redirect()->to(route(config('basket.quick_buy_redirect_route'))) : [
                'status' => 'redirect',
                'url' => route(config('basket.quick_buy_redirect_route')),
            ];
        }

        if ($request->wantsJson()) {
            return $response;
        }

        return redirect()->to($url);
    }

    /**
     * Add Items in basket from quick buy
     *
     * @param \Item $item
     * @param array $sizes
     */
    protected function putSizesInBasket(\Item $item, $sizes)
    {
        collect($sizes)
            ->filter()
            ->each(function ($quantity, $size_id) use ($item) {
                $size = Size::query()->find($size_id);
                if ($size) {
                    App::make(BasketRepositoryInterface::class)->addItem($size, $quantity);
                } else {
                    App::make(BasketRepositoryInterface::class)->addItem($item->eloquent(), $quantity);
                }
            });

        // Add a log of how many times it's added to the basket
        $item->incrementCount('num_basket');
    }
}
