<?php

namespace Mtc\BasketRecovery\Console\Commands;

use Illuminate\Console\Command;
use Mtc\Basket\Basket;
use Mtc\BasketRecovery\Models\BasketRecovery;
use Mtc\BasketRecovery\Models\BasketRecoveryEmail;

/**
 * Class NotifyAbandonedBaskets
 *
 * @package Mtc\BasketRecovery\Console
 */
class NotifyAbandonedBasketsCommand extends Command
{

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'basket-recovery:notify';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Notify users about their basket.';

    /**
     * Run the command
     *
     * @return void
     */
    public function handle()
    {
        $this->getActiveBasketRecoveries()
            ->each(function (BasketRecovery $basket_recovery) {
                $this->notifyCustomers($basket_recovery);
            });

    }

    /**
     * Find recoveries that are active
     *
     * @return BasketRecovery[]|\Illuminate\Database\Eloquent\Collection
     */
    protected function getActiveBasketRecoveries()
    {
        return BasketRecovery::query()
            ->where('is_active', true)
            ->get();
    }

    /**
     * Find and notify customers about their baskets
     *
     * @param \Mtc\BasketRecovery\Models\BasketRecovery $basket_recovery
     * @throws \Exception
     */
    protected function notifyCustomers(BasketRecovery $basket_recovery)
    {
        $this->info('checking users to notify');
        $basket_recovery->getNotifiableBaskets()
            ->each(function (Basket $basket) use ($basket_recovery) {
                $this->notifyBasket($basket, $basket_recovery);
            });
    }

    /**
     * @param \Mtc\Basket\Basket $basket
     * @param \Mtc\BasketRecovery\Models\BasketRecovery $basket_recovery
     *
     * @throws \Exception
     */
    protected function notifyBasket(Basket $basket, BasketRecovery $basket_recovery)
    {
        $coupon = $basket_recovery->generateCoupon();
        $email = BasketRecoveryEmail::create($basket, $basket_recovery);
        $items = $basket->items
            ->filter(function ($item) {
                // Make sure we don't send removed items
                return $item->purchasable;
            });

        $email_body = template('BasketRecovery/emails/basket_recovery_email.twig', [
            'basket_recovery' => $basket_recovery,
            'items' => $items,
            'currency' => $basket->unpaidOrders()->latest()->first()->currency_code ?? config('currencies.default_currency', ''),
            'coupon' => $coupon,
            'basket_recovery_url' => route('basket-recovery.recover', ['token' => $email->token]),
            'basket_recovery_image' => route('basket-recovery.read', ['token' => $email->token])
        ]);

        $this->info('notifying user');
        email($basket->email, $basket_recovery->subject, $email_body);
    }
}
