<?php

namespace Mtc\BasketRecovery\Queries;

use Carbon\Carbon;
use Illuminate\Database\Query\Builder;
use Mtc\Basket\Basket;
use Mtc\BasketRecovery\Models\BasketRecovery;
use Mtc\BasketRecovery\Models\BasketRecoveryEmail;
use Mtc\Orders\Order;

/**
 * Class RecoverableBasketsQuery
 *
 * @package Mtc\BasketRecovery\Queries
 */
class RecoverableBasketsQuery
{

    /**
     * @var \Mtc\BasketRecovery\Models\BasketRecovery
     */
    protected $basket_recovery;

    /**
     * AvailableBasketRecoveryOrders constructor.
     *
     * @param \Mtc\BasketRecovery\Models\BasketRecovery $basket_recovery
     */
    public function __construct(BasketRecovery $basket_recovery)
    {
        $this->basket_recovery = $basket_recovery;
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection|Basket[]
     * @throws \Exception
     */
    public function get()
    {
        return $this->builder()->get();
    }

    /**
     * The goal of this query is to retrieve a list of orders where the user has said, notify me AND the user
     * has not paid.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     * @throws \Exception
     */
    public function builder()
    {
        /*
         * If the basket has been paid in another order instance, we need to make sure that this not being brought back.
         * To do this we must get all of the orders which have been paid, grab their basket and return them.
         */
        $orders = Order::query()
            ->whereDoesntHave('paidSibling')
            ->where('created_at', 'like', $this->getBasketRecoveryDate()->format('Y-m-d H:i%'))
            ->groupBy("basket_id")
            ->pluck("basket_id");

        /*
         * If the user has already been notified about this basket, then we need to ensure that they are not sent
         * anymore emails as this will get spammy.
         */
        $already_notified = BasketRecoveryEmail::query()
            ->groupBy('basket_id')
            ->whereIn('basket_id', $orders)
            ->pluck('basket_id');

        $notifiable_orders = $orders->diff($already_notified->toArray());

        return Basket::query()
            ->whereNotNull('email')
            ->whereHas('newsletterSignUp')
            ->whereIn('id', $notifiable_orders)
            ->whereHas('items');
    }

    /**
     * Return the Y-m-d H:i format of the interval from the current time.
     *
     * @return Carbon
     * @throws \Exception
     */
    protected function getBasketRecoveryDate()
    {
        return Carbon::now()->sub($this->basket_recovery->getDateInterval());
    }
}
