<?php namespace Mtc\BasketRecovery\Models;


use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Mtc\BasketRecovery\Queries\RecoverableBasketsQuery;
use Mtc\Coupons\Coupon;

/**
 * Class BasketRecovery
 *
 * @property integer $id
 * @property string $subject
 * @property string $content
 * @property array|null $coupon_data
 * @property boolean $is_active
 * @property string $interval
 * @property integer $interval_seconds
 * @property-read string $interval_unit
 * @property-read string $interval_measurement
 * @property-read \Illuminate\Database\Eloquent\Collection|\Mtc\BasketRecovery\Models\BasketRecoveryLog[] $logs
 *
 * @package Mtc\BasketRecovery\Models
 */
class BasketRecovery extends Model
{

    /**
     * MySQL's permitted intervals.
     *
     * @var array
     */
    protected static $intervals = [
        'MINUTE',
        'HOUR',
        'DAY',
        'WEEK',
        'MONTH'
    ];

    /**
     * @var string
     */
    protected $table = 'basket_recoveries';

    /**
     * @var array
     */
    protected $fillable = [
        'subject',
        'content',
        'coupon_data',
        'is_active',
        'interval',
        'interval_seconds'
    ];

    /**
     * @var array
     */
    protected $casts = [
        'coupon_data' => 'array',
        'is_active' => 'boolean',
        'interval_seconds' => 'integer',
    ];

    /**
     * @var array
     */
    protected $attributes = [
        'is_active' => false
    ];

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function logs()
    {
        return $this->hasMany(BasketRecoveryLog::class);
    }


    /**
     * Set the interval for the email to run.
     *
     * @param int $unit Must be an integer greater than 0.
     * @param string $measure Must be a value contained within BasketRecovery::getAllowedIntervals()
     *
     * @return $this
     */
    public function setInterval($unit, $measure)
    {

        $unit = (int)$unit;

        if (!in_array($measure, $this->getAllowedIntervals())) {
            throw new \InvalidArgumentException("Measure not permitted: {$measure}");
        }

        if ($unit < 1) {
            throw new \InvalidArgumentException("Unit must be greater than 0");
        }

        $this->interval = "{$unit} {$measure}";
        $this->interval_seconds = strtotime("+{$this->interval}") - time();

        return $this;

    }

    /**
     * Return a list of permitted intervals.
     *
     * @return array
     */
    public static function getAllowedIntervals()
    {
        $intervals = [];
        foreach (self::$intervals as $interval) {
            $intervals[$interval] = $interval;
        }
        return $intervals;
    }

    /**
     * @return array
     */
    public function getAllowedUnits()
    {
        $allowed_units = [];
        for ($i = 1; $i <= 120; $i++) {
            $allowed_units[$i] = $i;
        }
        return $allowed_units;
    }

    /**
     * @return \DateInterval
     * @throws \Exception
     */
    public function getDateInterval()
    {
        return new \DateInterval("PT{$this->interval_seconds}S");
    }

    /**
     * @return \Illuminate\Database\Eloquent\Collection|\Mtc\Basket\Basket[]
     * @throws \Exception
     */
    public function getNotifiableBaskets()
    {
        return (new RecoverableBasketsQuery($this))->get();
    }

    /**
     * @return string|null
     */
    public function getIntervalUnitAttribute()
    {
        $exploded_interval = explode(' ', $this->interval);
        return $exploded_interval[0] ?? null;
    }

    /**
     * @return string|null
     */
    public function getIntervalMeasurementAttribute()
    {
        $exploded_interval = explode(' ', $this->interval);
        return $exploded_interval[1] ?? null;
    }

    /**
     * Returns an array of statistics for this basket recovery email.
     *
     * @return array|boolean
     */
    public function getStatistics()
    {
        if (!config('basket_recovery.statistics.enabled', false) || !$this->exists) {
            return false;
        }

        return [
            'read' => $this->getReadCount(),
            'clicked' => $this->getClickedCount(),
            'sales' => $this->getSalesCount(),
            'sales_value' => number_format($this->getSalesValue(), 2),
        ];

    }

    /**
     * @return int
     */
    public function getReadCount(): int
    {
        return $this->logs()->where('action', 'read')->count();
    }

    /**
     * @return int
     */
    public function getClickedCount(): int
    {
        return $this->logs()->where('action', 'click')->count();
    }

    /**
     * @return int
     */
    public function getSalesCount(): int
    {
        return $this->logs()->where('action', 'sale')->count();
    }

    /**
     * @return float
     */
    public function getSalesValue(): float
    {
        return (float)$this->logs()->where('action', 'sale')->sum('value');
    }

    /**
     * Generates the associated coupon for this email, if active and exists
     *
     * @return Coupon|bool|Model
     * @throws \Exception
     */
    public function generateCoupon()
    {

        if (config('basket_recovery.coupons.enabled', false) !== true) {
            return false;
        }

        if (is_null($this->coupon_data)) {
            return false;
        }

        if (!isset($this->coupon_data['type']) || empty($this->coupon_data['type'])) {
            return false;
        }

        if (empty($this->coupon_data['free_delivery']) && (float)$this->coupon_data['value'] <= 0) {
            return false;
        }

        $start_date = now();
        $expiry_date = now()->add(new \DateInterval(config('basket_recovery.coupons.expiry', 'P1W')));

        return Coupon::query()
            ->create([
                'name' => 'Recovered Basket Coupon - ' . $this->subject,
                'code' => $this->generateCouponCode(),
                'valid_from' => $start_date,
                'valid_to' => $expiry_date,
                'type' => $this->coupon_data['type'],
                'value' => $this->coupon_data['value'],
                'redemptions' => 1,
                'basic_restrictions' => [
                    'only_first_purchase' => 1,
                    'minimum_basket_price' => $this->coupon_data['min_price'],
                ],
            ]);

    }

    /**
     * @return string
     */
    protected function generateCouponCode()
    {
        $code = strtoupper(Str::random(10));

        if (!empty($prefix = config('basket_recovery.coupons.prefix', false))) {
            $code = "{$prefix}-{$code}";
        }

        if (Coupon::query()->where('code', $code)->exists()) {
            return $this->generateCouponCode();
        }

        return $code;
    }
}
