<?php

namespace Tests\Tenant;

use App\Facades\Feature;
use App\Facades\Settings;
use Illuminate\Support\Facades\Queue;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\KeyloopLeaseResidualValue;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\Label;
use Mtc\MercuryDataModels\LeaseVehicleLabel;
use Mtc\MercuryDataModels\TransmissionType;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Tests\LeaseFilterSettingSeed;
use Tests\TenantTestCase;

class LeaseVehicleFiltersTest extends TenantTestCase
{
    public function setUp(): void
    {
        parent::setUp();
        Queue::fake();
    }

    public function testMissingPrice()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        KeyloopLeaseVehicleVariant::factory(3)->create([
            'is_cheapest_variant' => true,
            'price_ex_vat_ex_vrt' => 0,
            'cheapest_monthly_price_ex_vat' => 123,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);
        $this->assertEquals(0, $response->json('results.total'));
    }

    public function testMissingMonthlyPrice()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        KeyloopLeaseVehicleVariant::factory(3)->create([
            'is_cheapest_variant' => true,
            'price_ex_vat_ex_vrt' => 99000,
            'cheapest_monthly_price_ex_vat' => 0,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);
        $this->assertEquals(0, $response->json('results.total'));
    }

    public function testFilterByMake()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);
        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();

        $cheapest_1 = KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => 1,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory(4)->create([
            'make_id' => $make->id,
            'model_id' => 1,
            'is_cheapest_variant' => false,
        ]);

        $cheapest_2 = KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make2->id,
            'model_id' => 2,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory(2)->create([
            'make_id' => $make2->id,
            'model_id' => 2,
            'is_cheapest_variant' => false,
        ]);

        $cheapest_3 = KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => 100,
            'model_id' => 3,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory(9)->create([
            'make_id' => 100,
            'model_id' => 3,
            'is_cheapest_variant' => false,
        ]);

        // No makes specified
        $response = $this
            ->postJson(route('lease-vehicles.index'));
        $response->assertStatus(200);

        // assert that we got one variant per model
        $this->assertEquals(3, $response->json('results.total'));

        // assert that each returned variant is cheapest variant for that model
        $this->assertContains(
            $response->json('results.data.0.id'),
            [
                $cheapest_1->id,
                $cheapest_2->id,
                $cheapest_3->id,
            ],
        );

        $this->assertContains(
            $response->json('results.data.1.id'),
            [
                $cheapest_1->id,
                $cheapest_2->id,
                $cheapest_3->id,
            ],
        );

        $this->assertContains(
            $response->json('results.data.2.id'),
            [
                $cheapest_1->id,
                $cheapest_2->id,
                $cheapest_3->id,
            ],
        );

        // First Make
        $response2 = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->slug,
                    ],
                ]
            ]);
        $response2->assertStatus(200);
        $this->assertEquals(1, $response2->json('results.total'));

        // Second make
        $response3 = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make2->slug,
                    ],
                ]
            ]);
        $response3->assertStatus(200);
        $this->assertEquals(1, $response3->json('results.total'));

        // Both Makes
        $response4 = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->slug,
                    ],
                    [
                        'type' => 'make',
                        'value' => $make2->slug,
                    ],
                ]
            ]);
        $response4->assertStatus(200);
        $this->assertEquals(2, $response4->json('results.total'));
    }

    public function testFilterResultOptions()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();
        $make3 = VehicleMake::factory()->create();

        KeyloopLeaseVehicleVariant::factory(5)->create([
            'make_id' => $make->id,
        ]);

        KeyloopLeaseVehicleVariant::factory(3)->create([
            'make_id' => $make2->id,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertCount(2, $response->json('filters.make.results'));
    }

    public function testAvailableModels()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();

        $model = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);

        $model2 = VehicleModel::factory()->create([
            'make_id' => $make2->id,
        ]);

        KeyloopLeaseVehicleVariant::factory(3)->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => false,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory(5)->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'is_cheapest_variant' => false,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'is_cheapest_variant' => true,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertCount(
            4,
            KeyloopLeaseVehicleVariant::query()
                ->where('model_id', '=', $model->id)
                ->get()
        );

        $this->assertCount(
            6,
            KeyloopLeaseVehicleVariant::query()
                ->where('model_id', '=', $model2->id)
                ->get()
        );

        $this->assertEquals(2, $response->json('results.total'));
    }

    public function testFilterByModel()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();

        $model = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);

        $model2 = VehicleModel::factory()->create([
            'make_id' => $make2->id,
        ]);

        KeyloopLeaseVehicleVariant::factory(3)->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => false,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory(5)->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'is_cheapest_variant' => false,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'is_cheapest_variant' => true,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->slug,
                    ],
                    [
                        'type' => 'model',
                        'value' => $model->slug,
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertCount(
            4,
            KeyloopLeaseVehicleVariant::query()
                ->where('model_id', '=', $model->id)
                ->get()
        );

        $this->assertCount(
            6,
            KeyloopLeaseVehicleVariant::query()
                ->where('model_id', '=', $model2->id)
                ->get()
        );

        // assert that we got the expected model variants
        $this->assertEquals(4, $response->json('results.total'));

        // assert that the cheapest variant filter should be hidden
        $this->assertArrayHasKey('make', $response->json('filters'));
        $this->assertArrayHasKey('cheapest_lease_variant', $response->json('filters'));
        $this->assertTrue($response->json('filters.make.show'));
        $this->assertFalse($response->json('filters.cheapest_lease_variant.show'));
    }

    public function testFilterByType()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        KeyloopLeaseVehicleVariant::factory(5)->create([
            'type' => 'car',
            'is_cheapest_variant' => true,
        ]);
        KeyloopLeaseVehicleVariant::factory(3)->create([
            'type' => 'lcv',
            'is_cheapest_variant' => true,
        ]);
        KeyloopLeaseVehicleVariant::factory(10)->create([
            'type' => null,
            'is_cheapest_variant' => true,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertEquals(18, $response->json('results.total'));

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'vehicle_type',
                        'value' => 'car',
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertEquals(5, $response->json('results.total'));

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'vehicle_type',
                        'value' => 'car',
                    ],
                    [
                        'type' => 'vehicle_type',
                        'value' => 'lcv',
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertEquals(8, $response->json('results.total'));

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'vehicle_type',
                        'value' => 'foo',
                    ],
                ],
            ]);

        $response->assertStatus(200);

        $this->assertEquals(0, $response->json('results.total'));
    }

    public function testResponseData()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'automotive-vehicle-model-on-filter-card',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'imagin-placeholders-lease-vehicle-enabled',
            'value' => true,
            'type' => 'bool',
        ]);

        Feature::setEnabled(tenant(), ['imagin-studio-placeholders']);

        $make = VehicleMake::factory()->create([
            'name' => 'foo',
        ]);
        $model = VehicleModel::factory()->create([
            'name' => 'bar',
        ]);
        $fuel = FuelType::factory()->create();
        $transmission = TransmissionType::factory()->create();
        $body_type = BodyStyleType::factory()->create();

        KeyloopLeaseVehicleVariant::factory()->create([
            'type' => 'car',
            'is_cheapest_variant' => true,
            'external_variant_id' => 'foo',
            'name' => 'foo bar turbo',
            'make_id' => $make->id,
            'model_id' => $model->id,
            'fuel_type_id' => $fuel->id,
            'transmission_id' => $fuel->id,
            'body_style_id' => $fuel->id,
            'price_ex_vat_ex_vrt' => 123,
            'price_inc_vat_inc_vrt' => 555,
            'vehicle_vrt' => 10,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertEquals(1, $response->json('results.total'));

        $this->assertArrayHasKey('external_variant_id', $response->json('results.data.0'));
        $this->assertArrayHasKey('make', $response->json('results.data.0'));
        $this->assertArrayHasKey('model', $response->json('results.data.0'));
        $this->assertArrayHasKey('derivative', $response->json('results.data.0'));
        $this->assertArrayHasKey('monthly_from_price_ex_vat', $response->json('results.data.0'));
        $this->assertArrayHasKey('monthly_from_price_inc_vat', $response->json('results.data.0'));
        $this->assertArrayHasKey('fuel_type', $response->json('results.data.0'));
        $this->assertArrayHasKey('transmission', $response->json('results.data.0'));
        $this->assertArrayHasKey('body_type', $response->json('results.data.0'));
        $this->assertArrayHasKey('imagin_studio_base_url', $response->json('results.data.0'));
        $this->assertArrayHasKey('labels', $response->json('results.data.0'));
        $this->assertArrayHasKey('colours', $response->json('results.data.0'));

        $this->assertEquals('foo', $response->json('results.data.0.external_variant_id'));
        $this->assertEquals($make->name, $response->json('results.data.0.make.name'));
        $this->assertEquals($model->name, $response->json('results.data.0.model.name'));
        $this->assertEquals('turbo', $response->json('results.data.0.derivative'));
        //$this->assertEquals('from_price_monthly', $response->json('results.data.0.from_price_monthly'));
        $this->assertEquals($fuel->name, $response->json('results.data.0.fuel_type'));
        $this->assertEquals($transmission->name, $response->json('results.data.0.transmission'));
        $this->assertEquals($body_type->name, $response->json('results.data.0.body_type'));
        $this->assertStringContainsString('cdn.imagin.studio/getImage', $response->json('results.data.0.imagin_studio_base_url'));
        $this->assertEquals(555, $response->json('results.data.0.rrp'));

        $this->assertArrayHasKey('finance_settings', $response->json('results'));
        $this->assertArrayHasKey('contract_terms', $response->json('results.finance_settings'));
        $this->assertArrayHasKey('deposit_quantity_months', $response->json('results.finance_settings'));
        $this->assertArrayHasKey('annual_mileages', $response->json('results.finance_settings'));

        $this->assertContains(24, $response->json('results.finance_settings.contract_terms'));
        $this->assertContains(3, $response->json('results.finance_settings.deposit_quantity_months'));
        $this->assertContains(15000, $response->json('results.finance_settings.annual_mileages'));
    }

    public function testFilterByFuel()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        $fuel_type_one = FuelType::factory()->create();
        $fuel_type_two = FuelType::factory()->create();

        $make = VehicleMake::factory()->create();
        $make2 = VehicleMake::factory()->create();

        $model = VehicleModel::factory()->create([
            'make_id' => $make->id,
        ]);

        $model2 = VehicleModel::factory()->create([
            'make_id' => $make2->id,
        ]);

        // this vehicle should match the search filters but is not the cheapest variant of this model
        KeyloopLeaseVehicleVariant::factory(3)->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => false,
            'fuel_type_id' => $fuel_type_one->id,
        ]);

        // this variant is the cheapest for the model, but should not  match the search filters
        $cheapest_variant = KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model->id,
            'is_cheapest_variant' => true,
            'fuel_type_id' => $fuel_type_two->id,
        ]);

        // these vehicles match the fuel type, but don't match the make or model
        KeyloopLeaseVehicleVariant::factory(5)->create([
            'make_id' => $make2->id,
            'model_id' => $model2->id,
            'is_cheapest_variant' => false,
            'fuel_type_id' => $fuel_type_one->id,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'make',
                        'value' => $make->slug,
                    ],
                    [
                        'type' => 'fuel_type',
                        'value' => $fuel_type_one->slug,
                    ],
                ],
            ]);

        $response->assertStatus(200);
        $this->assertEquals(1, $response->json('results.total'));

        // we expect the search to return the cheapest variant of the model which our matching variant belongs to
        $this->assertEquals($cheapest_variant->id, $response->json('results.data.0.id'));
    }

    public function testFinanceOptions()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Enabled',
            'config_key' => 'leasing-keyloop-fleet-kompact-enabled',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Provider Type',
            'config_key' => 'leasing-keyloop-fleet-kompact-provider-type',
            'type' => 'string',
            'value' => 'kearys',
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'VAT percentage in decimal',
            'description' => 'For use in finance calculations. Decimal representation of a percentage. 7% should be entered as 0.07',
            'config_key' => 'leasing-keyloop-fleet-kompact-vat-rate',
            'type' => 'number',
            'value' => 0.23,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Leasing',
            'group' => 'Keyloop Fleet Kompact',
            'name' => 'Interest rate percentage in decimal',
            'description' => 'For use in finance calculations. Decimal representation of a percentage. 7% should be entered as 0.07',
            'config_key' => 'leasing-keyloop-fleet-kompact-interest-rate',
            'type' => 'number',
            'value' => 0.07,
        ]);

        $variant = KeyloopLeaseVehicleVariant::factory()->create([
            'is_cheapest_variant' => true,
            'price_ex_vat_ex_vrt' => 34766.33,
            'delivery_ex_vat_ex_vrt' => 0,
            'vehicle_vrt' => 0,
            'finance_recently_updated' => false,
            'cheapest_monthly_price_ex_vat' => 123,
            'cheapest_monthly_price_inc_vat' => 200,
        ]);

        KeyloopLeaseResidualValue::factory()->create([
            'external_variant_id' => $variant->external_variant_id,
            'contract_length' => 36,
            'annual_mileage' => 15000,
            'residual_value_excluding_vat' => 11871.43,
            'residual_value_including_vat' => 11871.43 * 1.23,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertEquals(1, $response->json('results.total'));
        $variant = $response->json('results.data.0');

        $this->assertArrayHasKey('finance_options', $variant);
        $this->assertEquals(732.24, $variant['finance_options']['monthly_prices']['36-15000-12']['monthly_price_ex_vat']);
        $this->assertEquals(732.24, $variant['finance_options']['cheapest_monthly_price']['monthly_price_ex_vat']);
    }

    public function testSorting()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-latest',
            'type' => 'boolean',
            'value' => true,
            'order' => 5,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-lease-vehicle-price-asc',
            'type' => 'boolean',
            'value' => true,
            'order' => 5,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-lease-vehicle-price-desc',
            'type' => 'boolean',
            'value' => true,
            'order' => 1,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-lease-vehicle-monthly-price-asc',
            'type' => 'boolean',
            'value' => true,
            'order' => 5,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Sorting',
            'name' => 'Price (Low > High)',
            'config_key' => 'automotive-vehicle-sorting-lease-vehicle-monthly-price-desc',
            'type' => 'boolean',
            'value' => true,
            'order' => 5,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 111,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 333,
            'is_cheapest_variant' => true,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'price_ex_vat_ex_vrt' => 222,
            'is_cheapest_variant' => true,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'sort_by' => 'lease-vehicle-price-desc',
            ]);

        $response->assertStatus(200);
        $this->assertCount(5, $response->json('sort_options'));
        $this->assertEquals('lease-vehicle-price-desc', $response->json('sort_options.0.code'));
        $this->assertCount(3, $response->json('results.data'));
        $this->assertEquals(333, $response->json('results.data.0.price_ex_vat_ex_vrt'));
    }

    public function testLabels()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        Settings::make([
            'tab' => 'foo',
            'section' => 'lorem',
            'group' => 'ipsum',
            'name' => 'dolor',
            'config_key' => 'lease-vehicle-labels-enabled',
            'value' => true,
            'type' => 'boolean',
        ]);

        $label_applicable = Label::factory()->create([
            'name' => 'applicable',
            'type' => 'all'
        ]);

        $vehicle_applicable = KeyloopLeaseVehicleVariant::factory()->create([
            'is_cheapest_variant' => true,
        ]);

        LeaseVehicleLabel::create([
            'vehicle_id' => $vehicle_applicable->id,
            'label_id' => $label_applicable->id
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);

        $this->assertEquals(1, $response->json('results.total'));
        $this->assertCount(1, LeaseVehicleLabel::all());
        $this->assertNotNull($response->json('results.data.0.labels'));
        $this->assertCount(1, $response->json('results.data.0.labels'));
    }

    public function testPriceFilter()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        // this vehicle should match the search filters but is not the cheapest variant of this model
        KeyloopLeaseVehicleVariant::factory(3)->create([
            'is_cheapest_variant' => true,
            'price_ex_vat_ex_vrt' => 123,
        ]);

        KeyloopLeaseVehicleVariant::factory(2)->create([
            'is_cheapest_variant' => true,
            'price_ex_vat_ex_vrt' => 456,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price_min',
                        'value' => 111,
                    ],
                    [
                        'type' => 'price_max',
                        'value' => 222,
                    ],
                ],
            ]);

        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('results.total'));
    }

    public function testMonthlyPriceFilter()
    {
        $this->seed();
        $this->seed(LeaseFilterSettingSeed::class);

        // this vehicle should match the search filters but is not the cheapest variant of this model
        KeyloopLeaseVehicleVariant::factory(3)->create([
            'is_cheapest_variant' => true,
            'cheapest_monthly_price_ex_vat' => 123,
        ]);

        KeyloopLeaseVehicleVariant::factory(2)->create([
            'is_cheapest_variant' => true,
            'cheapest_monthly_price_ex_vat' => 456,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'), [
                'selections' => [
                    [
                        'type' => 'price_monthly_min',
                        'value' => 111,
                    ],
                    [
                        'type' => 'price_monthly_max',
                        'value' => 222,
                    ],
                ],
            ]);

        $response->assertStatus(200);
        $this->assertEquals(3, $response->json('results.total'));
    }

    public function testFilterOptionsIgnoreActive()
    {
        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Make',
            'config_key' => 'automotive-vehicle-filters-make',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Model',
            'config_key' => 'automotive-vehicle-filters-model',
            'type' => 'boolean',
            'value' => true,
        ]);

        $make = VehicleMake::factory()->create();
        $model_one = VehicleModel::factory()->create(['make_id' => $make->id]);
        $model_two = VehicleModel::factory()->create(['make_id' => $make->id]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_one->id,
            'cheapest_monthly_price_ex_vat' => 0,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_two->id,
            'cheapest_monthly_price_ex_vat' => 123,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);
        $this->assertCount(1, $response->json('filters.make.results'));
        $this->assertCount(2, $response->json('filters.model.results'));
    }

    public function testFilterOptionsUseActive()
    {
        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Use active scope for available filter options',
            'config_key' => 'filter-options-use-active-scope',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Make',
            'config_key' => 'automotive-vehicle-filters-make',
            'type' => 'boolean',
            'value' => true,
        ]);

        Settings::make([
            'tab' => 'Automotive',
            'section' => 'Filter',
            'group' => 'Filtering',
            'name' => 'Model',
            'config_key' => 'automotive-vehicle-filters-model',
            'type' => 'boolean',
            'value' => true,
        ]);

        $make = VehicleMake::factory()->create();
        $model_one = VehicleModel::factory()->create(['make_id' => $make->id]);
        $model_two = VehicleModel::factory()->create(['make_id' => $make->id]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_one->id,
            'cheapest_monthly_price_ex_vat' => 0,
        ]);

        KeyloopLeaseVehicleVariant::factory()->create([
            'make_id' => $make->id,
            'model_id' => $model_two->id,
            'cheapest_monthly_price_ex_vat' => 123,
        ]);

        $response = $this
            ->postJson(route('lease-vehicles.index'));

        $response->assertStatus(200);
        $this->assertCount(1, $response->json('filters.make.results'));
        $this->assertCount(1, $response->json('filters.model.results'));
        $this->assertEquals($model_two->slug, $response->json('filters.model.results.0.slug'));
    }
}
