<?php

namespace App\Traits;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\SeoData;
use Mtc\MercuryDataModels\SeoDefault;

trait RetrievesSeoData
{
    use CacheObject;

    protected Collection $mediaSizes;

    private function fillSeo(string $section, ?string $urlPath = null): array
    {
        if (!empty($urlPath)) {
            $data = $this->getSeoForPath($urlPath);
            if ($data) {
                return $data->only([
                    'title',
                    'description',
                    'heading',
                    'content',
                    'canonical',
                ]);
            }
        }

        $seoData = $this->resource['seo'] ?? [];
        if (!empty($seoData['title']) && !empty($seoData['description'])) {
            return $seoData;
        }


        $default = $this->cache(
            'seo-default-' . $section,
            15,
            fn() => SeoDefault::query()->where('section', $section)->first()
        );
        if (!$default) {
            return $seoData;
        }

        if (empty($seoData['title'])) {
            $seoData['title'] = $this->replaceSeoTags([
                '{{TITLE}}' =>  $this->resource->title ?? $this->resource->name,
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{TRANSMISSION}}' => $this->resource->transmission->name ?? '',
                '{{MANUFACTURE_YEAR}}' => $this->resource->manufacture_year ?? '',
                '{{COLOUR}}' => $this->resource->colour ?? '',
                '{{VEHICLE_TYPE}}' => $this->resource->vehicle_type ?? '',
                '{{CONDITION}}' => $this->getConditionValue(),
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{NEW_USED}}' => ($this->resource->is_new ? __('labels.vehicle.new') : __('labels.vehicle.used')),

            ], $default->title ?? $this->resource->title);
        }

        if (empty($seoData['description'])) {
            $seoData['description'] = $this->replaceSeoTags([
                '{{TITLE}}' =>  $this->resource->title ?? $this->resource->name,
                '{{SITE_NAME}}' => Settings::get('app-name'),
                '{{MAKE}}' => $this->resource->make->name ?? '',
                '{{MODEL}}' => $this->resource->model->name ?? '',
                '{{FUEL_TYPE}}' => $this->resource->fuelType->name ?? '',
                '{{BODY_STYLE}}' => $this->resource->bodyStyle->name ?? '',
                '{{TRANSMISSION}}' => $this->resource->transmission->name ?? '',
                '{{MANUFACTURE_YEAR}}' => $this->resource->manufacture_year ?? '',
                '{{COLOUR}}' => $this->resource->colour ?? '',
                '{{VEHICLE_TYPE}}' => $this->resource->vehicle_type ?? '',
                '{{CONDITION}}' => $this->getConditionValue(),
                '{{CONTENT_EXCERPT}}' => Str::limit($this->resource->description, 150),
                '{{NEW_USED}}' => ($this->resource->is_new ? __('labels.vehicle.new') : __('labels.vehicle.used')),

            ], $default->description ?? '');
        }
        return $seoData;
    }

    protected function getConditionValue()
    {
        if ($this->resource->is_new) {
            return __('labels.vehicle.new');
        }

        return $this->resource->ex_demo ? __('labels.vehicle.ex-demo') : __('labels.vehicle.used');
    }

    private function replaceSeoTags(array $replacements, string $string): string
    {
        return str_replace(array_keys($replacements), $replacements, $string);
    }

    private function getSeoForPath(string $path): ?Model
    {
        return SeoData::query()->where('path', $path)->first();
    }
}
