<?php

namespace Tests\Tenant;

use App\Modules\Stock\Modix;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleFeature;
use SimpleXMLElement;
use Tests\TenantTestCase;

class ModixTest extends TenantTestCase
{
    public function testDates()
    {
        $xml = '<vehicles>
                    <vehicle>
                        <dates>
                            <date id="2">2022-06-24</date>
                            <date id="1">2020-02-06</date>
                            <date id="7">2024-03-07</date>
                        </dates>
                    </vehicle>
                </vehicles>';

        $xml_element = new SimpleXMLElement($xml);
        $vehicles = $xml_element->xpath('//vehicles/vehicle');
        $vehicle = reset($vehicles);

        $modix = new Modix();
        $methodGetManufactureDate = new \ReflectionMethod(Modix::class, 'getManufactureDate');
        $methodGetFirstRegistrationDate = new \ReflectionMethod(Modix::class, 'getFirstRegistrationDate');
        $methodGetManufactureDate->setAccessible(true);
        $methodGetFirstRegistrationDate->setAccessible(true);
        $manufacture_date = $methodGetManufactureDate->invoke($modix, $vehicle);
        $registration_date = $methodGetFirstRegistrationDate->invoke($modix, $vehicle);

        $this->assertEquals('2020-02-06', $manufacture_date);
        $this->assertEquals('2022-06-24', $registration_date);
    }

    public function testDatesEmpty()
    {
        $xml = '<vehicles>
                    <vehicle>
                        <dates>
                        </dates>
                    </vehicle>
                </vehicles>';

        $xml_element = new SimpleXMLElement($xml);
        $vehicles = $xml_element->xpath('//vehicles/vehicle');
        $vehicle = reset($vehicles);

        $modix = new Modix();
        $methodGetManufactureDate = new \ReflectionMethod(Modix::class, 'getManufactureDate');
        $methodGetFirstRegistrationDate = new \ReflectionMethod(Modix::class, 'getFirstRegistrationDate');
        $methodGetManufactureDate->setAccessible(true);
        $methodGetFirstRegistrationDate->setAccessible(true);
        $manufacture_date = $methodGetManufactureDate->invoke($modix, $vehicle);
        $registration_date = $methodGetFirstRegistrationDate->invoke($modix, $vehicle);

        $this->assertEmpty($manufacture_date);
        $this->assertEmpty($registration_date);
    }

    public function testDatesMissing()
    {
        $xml = '<vehicles>
                    <vehicle>
                    </vehicle>
                </vehicles>';

        $xml_element = new SimpleXMLElement($xml);
        $vehicles = $xml_element->xpath('//vehicles/vehicle');
        $vehicle = reset($vehicles);

        $modix = new Modix();
        $methodGetManufactureDate = new \ReflectionMethod(Modix::class, 'getManufactureDate');
        $methodGetFirstRegistrationDate = new \ReflectionMethod(Modix::class, 'getFirstRegistrationDate');
        $methodGetManufactureDate->setAccessible(true);
        $methodGetFirstRegistrationDate->setAccessible(true);
        $manufacture_date = $methodGetManufactureDate->invoke($modix, $vehicle);
        $registration_date = $methodGetFirstRegistrationDate->invoke($modix, $vehicle);

        $this->assertEmpty($manufacture_date);
        $this->assertEmpty($registration_date);
    }

    public function testEvData()
    {
        $xml = '<vehicles>
                    <vehicle>
                        <colors>
                            <bodyColors>
                                <bodyColor id="7">
                                      <name>Grey</name>
                                      <description>Vapour Grey</description>
                                      <paint id="1">solid</paint>
                                </bodyColor>
                            </bodyColors>
                        </colors>
                        <prices>
                            <price type="2" currency="GBP">
                                <grossPrice>29850</grossPrice>
                            </price>
                            <price type="4" currency="GBP">
                                <grossPrice>29850</grossPrice>
                            </price>
                        </prices>
                        <dates>
                            <date id="1">2025-02-20</date>
                            <date id="2">2025-06-16</date>
                        </dates>
                        <media>
                            <images url="https://picserver.eu-central-1.eu.mdxprod.io/userdata/1/46973/3FAdVwEMPE/">
                                <meta>
                                    <image name="xxl_kfz_112075572_1761134746_1.jpg" position="1" updatedDateTime="2025-10-22 14:05:47">
                                        <tags>
                                            <tag id="3"/>
                                        </tags>
                                    </image>
                                </meta>
                                <image name="xxl_kfz_112075572_1761134746_1.jpg" position="1" updatedDateTime="2025-10-22 14:05:47"/>
                            </images>
                        </media>
                        <options>
                            <option id="8">
                                <value>2</value>
                                <name>Air conditioning</name>
                                <description>Air conditioning</description>
                            </option>
                            <option id="928">
                                <value>82</value>
                                <name>82.0 kWh</name>
                                <description>Battery capacity</description>
                            </option>
                            <option id="927">
                                <value>30</value>
                                <name>30 mi</name>
                                <description>NEDC electric range</description>
                            </option>
                        </options>
                    </vehicle>
                </vehicles>';

        $xml_element = new SimpleXMLElement($xml);
        $vehicles = $xml_element->xpath('//vehicles/vehicle');
        $vehicle = reset($vehicles);

        $modix = new Modix();
        $methodMapVehicleData = new \ReflectionMethod(Modix::class, 'mapVehicleData');
        $methodGetStockLocations = new \ReflectionMethod(Modix::class, 'getAllStockLocations');
        $methodMapVehicleData->setAccessible(true);
        $methodGetStockLocations->setAccessible(true);
        $methodGetStockLocations->invoke($modix, 'foo');
        $vehicle_data = $methodMapVehicleData->invoke($modix, $vehicle);

        $this->assertEquals(82, $vehicle_data['battery_capacity_kwh']);
        $this->assertEquals(82, $vehicle_data['battery_usable_capacity_kwh']);
        $this->assertEquals(30, $vehicle_data['battery_range']);
    }

    public function testFeatures()
    {
        $vehicle = Vehicle::factory()->create();
        VehicleFeature::factory()->create([
            'vehicle_type' => 'vehicle',
            'vehicle_id' => $vehicle->id,
            'name' => 'foo',
        ]);
        VehicleFeature::factory()->create([
            'vehicle_type' => 'vehicle',
            'vehicle_id' => $vehicle->id,
            'name' => 'baz',
        ]);
        VehicleFeature::factory()->create([
            'vehicle_type' => 'vehicle',
            'vehicle_id' => $vehicle->id,
            'name' => 'bar',
        ]);
        VehicleFeature::factory()->create([
            'vehicle_type' => 'vehicle',
            'vehicle_id' => $vehicle->id + 1,
            'name' => 'other',
        ]);

        $updated_features = [
            'foo',
            'buzz',
        ];

        $modix = new Modix();
        $methodSyncFeatures = new \ReflectionMethod(Modix::class, 'syncFeatures');
        $methodSyncFeatures->setAccessible(true);
        $methodSyncFeatures->invoke($modix, $vehicle, $updated_features);

        // assert that we didn't remove features for other vehicles
        $this->assertCount(1, VehicleFeature::query()->where('vehicle_id', '=', $vehicle->id + 1)->get());

        // assert that we removed features that don't exist in the incoming data
        $this->assertCount(2, VehicleFeature::query()->where('vehicle_id', '=', $vehicle->id)->get());
        $this->assertCount(1,  VehicleFeature::query()->where('vehicle_id', '=', $vehicle->id)
            ->where('name', 'foo')->get());
        $this->assertCount(1,  VehicleFeature::query()->where('vehicle_id', '=', $vehicle->id)
            ->where('name', 'buzz')->get());
    }
}
