<?php

namespace Tests\Tenant;

use App\Master\Models\VehicleMake;
use Mtc\MercuryDataModels\Franchise;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class FranchiseControllerTest extends TenantTestCase
{
    use UserForTenant;

     /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testIndex()
    {
        Franchise::factory(3)->create();

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.franchises.index'));

        $response->assertStatus(200);
        $this->assertIsArray($response->json('data'));
        $this->assertCount(3, $response->json('data'));
    }

    public function testStore()
    {
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.franchises.store'), [
                'name' => 'foo'
            ]);

        $response->assertStatus(201);
        $this->assertEquals('foo', $response->json('name'));
        $this->assertTrue(Franchise::query()->where('name', 'foo')->exists());

    }

    public function testUpdate()
    {
        $make = VehicleMake::factory()->create();
        $franchise = Franchise::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.franchises.update', $franchise->id), [
                'name' => 'baz',
                'make_id' => $make->id,
                'primary_colour' => 'A1B2C3',
                'secondary_colour' => 'AABBCC',
            ]);

        $franchise->refresh();
        $response->assertStatus(200);
        $this->assertEquals('AABBCC', $response->json('franchise.secondary_colour'));
        $this->assertEquals('AABBCC', $franchise->secondary_colour);
        $this->assertEquals('baz', $franchise->name);
        $this->assertEquals($make->id, $franchise->make_id);

    }

    public function testShow()
    {
        $franchise = Franchise::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.franchises.show', $franchise->id));

        $response->assertStatus(200);
        $this->assertEquals($franchise->id, $response->json('franchise.id'));
        $this->assertEquals('foo', $response->json('franchise.name'));
        $this->assertArrayHasKey('makes', $response->json());

    }

    public function testDelete()
    {
        $franchise = Franchise::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->deleteJson(route('tenant.franchises.destroy', $franchise->id));

        $response->assertStatus(200);
        $this->assertFalse(Franchise::query()->where('id', $franchise->id)->exists());
    }

    public function testCopy()
    {
        $franchise = Franchise::factory()->create(['name' => 'foo']);

        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->postJson(route('tenant.franchise.copy', $franchise->id), [
                'name' => 'baz bar',
            ]);

        $this->assertEquals(2, Franchise::query()->count());
        $this->assertNotEquals($franchise->id, $response->json('franchise.id'));
        $this->assertEquals('baz bar', $response->json('franchise.name'));
        $this->assertEquals($franchise->secondary_colour, $response->json('franchise.secondary_colour'));

    }
}
