<?php

namespace Tests\Tenant\Controllers;

use App\Tier;
use Database\Seeders\Tenant\ChatSettingSeeder;
use Illuminate\Support\Facades\Event;
use Mtc\MercuryDataModels\Events\SettingUpdated;
use Mtc\MercuryDataModels\Setting;
use Tests\TenantTestCase;
use Tests\UserForTenant;

class SettingControllerTest extends TenantTestCase
{
    use UserForTenant;

    protected bool $useTestSettingRepository = false;

    /**
     * A basic feature test example.
     *
     * @return void
     */
    public function testShow()
    {
        Setting::factory(10)->create(['tab' => 'foo']);
        $response = $this->actingAs($this->getUser())
            ->withoutExceptionHandling()
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.show', 'foo'));

        $response->assertStatus(200);

        $this->assertIsArray($response->json());
    }

    public function testIndex()
    {
        Setting::factory(10)->create(['tab' => 'foo']);
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.index'));

        $response->assertStatus(200);

        $this->assertIsArray($response->json());
    }

    public function testStore()
    {
        tenant()->tier = Tier::ENTERPRISE->value;
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.store'), [
                'tab' => 'foo-baz',
                'section' => 'lorem',
                'group' => 'ipsum',
                'name' => 'dolor',
                'config_key' => 'foo',
                'value' => 'baz',
                'type' => 'string',
            ]);

        $response->assertStatus(200);
        $this->assertIsArray($response->json());
    }

    public function testUpdate()
    {
        $setting = Setting::factory()->create([
            'section' => 'foo',
            'config_key' => 'foo.baz.bar',
            'value' => 'foo-baz-bar',
        ]);

        Event::fake();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->putJson(route('tenant.settings.update', 'foo'), [
                'settings' => [
                    [
                        'config_key' => 'foo.baz.bar',
                        'value' => 'Lorem Ipsum'
                    ]
                ]
            ]);
        $response->assertStatus(200);
        Event::assertDispatched(SettingUpdated::class);

        $setting->refresh();
        $this->assertEquals('Lorem Ipsum', $setting->value);
    }

    public function testSettingsAreGroupedAndOrdered()
    {
        (new ChatSettingSeeder())->run();
        $response = $this->actingAs($this->getUser())
            ->withHeader('X-Tenant', tenant('id'))
            ->getJson(route('tenant.settings.show', 'CRM'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('LiveChat', $response->json('0.data'));
        $this->assertEquals('chat-live-chat-enabled', $response->json('0.data.LiveChat.0.config_key'));
    }
}
