<?php

namespace App\Modules\Leasing;

use App\Modules\Leasing\Jobs\ImportVariantsJob;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\BodyStyleType;
use Mtc\MercuryDataModels\FuelType;
use Mtc\MercuryDataModels\KeyloopLeaseVehicleVariant;
use Mtc\MercuryDataModels\TransmissionType;

class KeyloopFleetKompactVariantImporter extends KeyloopFleetKompactImporterBase
{
    protected function filter(Collection $data): bool
    {
        if (
            array_key_exists($data['fran'], $this->enabled_keyloop_makes)
            && array_key_exists($data['fran'], $this->enabled_keyloop_models)
            && array_key_exists($data['model-code'], $this->enabled_keyloop_models[$data['fran']])
        ) {
            return true;
        }

        return false;
    }

    protected function importItems(Collection $data): void
    {
        $this->dispatch(new ImportVariantsJob($data->map(fn ($item) => [
            'external_variant_id' => $item['variant'],
            'was_recently_synced' => true,
            'make_id' => $this->getVehicleMake($item['fran']),
            'model_id' => $this->getVehicleModel($item['model-code'], $item['fran']),
            'transmission_id' => $this->getTransmissionTypeId($item['trans']),
            'fuel_type_id' => $this->getFuelTypeId($item['fuel'], $item['co2']),
            'body_style_id' => $this->getBodyTypeId($item['body-style']),
            'name' => $this->getUtf8String($item['description']),
            'is_current_variant' => $item['current-variant'] == 'Y',
            'type' => $this->getVehicleType($item['vehtype']),
            'engine_size_cc' => $item['engine-size'],
            'brake_horsepower' => $item['bhp'],
            'co2' => $item['co2'],
            'door_count' => $item['doors'],
            'seats' => $item['seats'],
            'road_fund_licence' => $item['rfl'],
            'finance_recently_updated' => false,
        ])));
    }

    protected function prepareForImport(): void
    {
        KeyloopLeaseVehicleVariant::query()->update([
            'was_recently_synced' => false,
        ]);
    }

    private function getTransmissionTypeId(string $transmission_code): ?int
    {
        $transmission_name = strtoupper($transmission_code) == 'M' ? 'Manual' : 'Automatic';
        return $this->transmission_types[$transmission_name] ?? null;
    }

    private function getFuelTypeId(string $fuel_code, int $co2): ?int
    {
        $fuel_type = '';

        switch ($fuel_code) {
            case 'D':
                $fuel_type = 'Diesel';
                break;
            case 'E':
                // the data can use 'E' for both hybrid and electric
                $fuel_type = $this->getSpecificElectricFuelType($co2);
                break;
            case 'H':
                $fuel_type = 'Hybrid';
                break;
            case 'P':
            case 'U':
                $fuel_type = 'Petrol';
                break;
            default:
                break;
        }

        return $this->fuel_types[$fuel_type] ?? null;
    }

    private function getSpecificElectricFuelType(int $co2): string
    {
        return empty($co2)
            ? 'Electric'
            : 'Hybrid';
    }

    private function getBodyTypeId(string $body_type): ?int
    {
        return $this->body_types[$body_type] ?? null;
    }

    private function getVehicleType(string $type): ?string
    {
        return match (strtolower($type)) {
            'car' => 'car',
            'comm' => 'lcv',
            default => null,
        };
    }
}
