<?php

namespace App\Jobs;

use App\NewCarRepository;
use App\OfferRepository;
use App\PageRepository;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\App;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\MercuryDataModels\NewCar;
use Mtc\MercuryDataModels\VehicleOffer;

class SyncContentToTemplate implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    private PageRepository $pageRepository;
    private OfferRepository $offerRepository;
    private NewCarRepository $newCarRepository;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(private readonly Model $model)
    {
        $this->onQueue('sync');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $this->loadRelationships();
        if (
            $this->model instanceof PageModel
            && $this->pageRepository()->isContentOutOfSync($this->model)
        ) {
            $this->pageRepository()->syncContentWithTemplate($this->model->id);
        }

        if (
            $this->model instanceof VehicleOffer
            && $this->offerRepository()->isContentOutOfSync($this->model)
        ) {
            $this->offerRepository()->syncContentWithTemplate($this->model->id);
        }

        if (
            $this->model instanceof NewCar
            && $this->newCarRepository()->isContentOutOfSync($this->model)
        ) {
            $this->newCarRepository()->syncContentWithTemplate($this->model->id);
        }
    }


    private function pageRepository(): PageRepository
    {
        if (!isset($this->pageRepository)) {
            $this->pageRepository = App::make(PageRepository::class);
        }

        return $this->pageRepository;
    }

    private function offerRepository(): OfferRepository
    {

        if (!isset($this->offerRepository)) {
            $this->offerRepository = App::make(OfferRepository::class);
        }

        return $this->offerRepository;
    }

    private function newCarRepository(): NewCarRepository
    {
        if (!isset($this->newCarRepository)) {
            $this->newCarRepository = App::make(NewCarRepository::class);
        }

        return $this->newCarRepository;
    }

    private function loadRelationships()
    {
        $this->model->load([
            'template.elements.element.fields',
            'content.mediaUses.media',
            'content.subContent.mediaUses.media',
            'content.subContent.subContent.mediaUses.media',
            'content.subContent.subContent.subContent.mediaUses.media',
            'content.globalContent.contentElement.fields',
            'content.contentElement.fields',
            'content.subContent.contentElement.fields',
            'content.subContent.subContent.contentElement.fields',
            'content.subContent.subContent.subContent.contentElement.fields',
        ]);
    }
}
