<?php

namespace App\Http\Resources;

use App\Traits\FormatAsCurrency;
use App\Traits\FranchiseData;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Resources\Json\JsonResource;
use Mtc\MercuryDataModels\NewCar;

class NewCarListResource extends JsonResource
{
    use FormatAsCurrency;

    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        if ($this->resource instanceof LengthAwarePaginator) {
            $this->resource->through(fn(NewCar $car) => $this->mapCar($car));

            $data = $this->resource->toArray();
            $data['view_columns'] = [
                [
                    'id' => 'make_model',
                    'name' => 'Make & Model',
                    'icon' => 'cars',
                ],
                [
                    'id' => 'price',
                    'name' => 'Price',
                    'icon' => 'tag',
                ],
                [
                    'id' => 'status',
                    'name' => 'Status',
                    'icon' => 'tag',
                ],
                [
                    'id' => 'enquiry_count',
                    'name' => 'Enquiry count',
                    'icon' => 'envelope-dot',
                ],
                [
                    'id' => 'days_since_update',
                    'name' => 'Days since last update',
                    'icon' => 'reply-clock',
                ],
            ];
            $data['soft_deletes'] = true;
            $data['default_columns'] = [
                'make_model',
                'price',
            ];
            return $data;
        }

        $this->resource = $this->resource->map(fn(NewCar $car) => $this->mapCar($car));

        return parent::toArray($request);
    }

    /**
     * Convert vehicle to the json format used in lists
     *
     * @param NewCar $car
     * @return array
     */
    protected function mapCar(NewCar $car)
    {
        return [
            'image' => [
                'src' => $car->getPreviewImage('mini-thumb'),
                'alt' => $car->name,
            ],
            'active' => (bool)$car->published,
            'status' => $car->status,
            'title' => $car->name,
            'description' => $car->derivative,
            'deleted_at' => $car->deleted_at,
            'link' => '/manage-content/new-vehicles/edit/' . $car->id,
            'metaItems' => [
                'make_model' => $car->make?->name . ' ' . $car->model?->name,
                'price' => $this->asCurrency($car->price ?? 0, tenant()->currency),
                'status' => $car->published ? 'Published' : 'Draft',
                'search_views' => $car->search_views ?? 0,
                'views' => $car->view_count ?? 0,
                'enquiry_count' => $car->enquiry_count ?? 0,
                'days_since_update' => $car->days_since_update,
                'hello'
            ],
            'id' => $car->id,
            'featured' => (bool) $car->featured,
        ];
    }
}
