<?php

namespace App\Http\Controllers;

use App\CarConfiguratorRepository;
use App\Http\Requests\CreateCarConfigurator;
use App\Http\Requests\SelectionSearchSearchRequest;
use App\Http\Resources\CarConfiguratorList;
use App\Http\Resources\CarConfiguratorView;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Str;
use Mtc\ContentManager\Http\Resources\MediaEntryResource;
use Mtc\MercuryDataModels\CarConfiguratorModel;
use Mtc\MercuryDataModels\MediaUse;
use Mtc\MercuryDataModels\VehicleMake;

class CarConfiguratorController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(SelectionSearchSearchRequest $request): CarConfiguratorList
    {
        $perPage = min((int) $request->input('per_page', 15), 200);
        $term = $request->input('search_term');
        return new CarConfiguratorList(CarConfiguratorModel::query()
            ->setSortBy($request->input('sort_by', 'id_desc'))
            ->when($request->filled('selections'), fn($query) => $query->setSelections($request->getSelections()))
            ->when($request->filled('search_term'), fn($query) => $query->where('name', 'like', "%$term%"))
            ->paginate($perPage));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateCarConfigurator $request, CarConfiguratorRepository $repository)
    {
        $configurator = $repository->create($request);
        return response($configurator);
    }

    /**
     * Display the specified resource.
     */
    public function show(CarConfiguratorModel $carConfigurator): CarConfiguratorView
    {
        return new CarConfiguratorView($carConfigurator);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(
        Request $request,
        CarConfiguratorModel $carConfigurator,
        CarConfiguratorRepository $repository
    ): CarConfiguratorView {
        $repository->update($request, $carConfigurator);
        $carConfigurator->refresh();
        return new CarConfiguratorView($carConfigurator);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(
        SelectionSearchSearchRequest $request,
        CarConfiguratorModel $carConfigurator,
        CarConfiguratorRepository $repository
    ): CarConfiguratorList {
        $repository->remove($carConfigurator);
        return $this->index($request);
    }

    public function getMedia(Request $request, CarConfiguratorModel $carConfigurator): Response
    {
        $query = $carConfigurator->mediaUses();
        $flags = collect($request->input('tags'))->map(fn($item) => Str::slug($item))->implode('-');
        $query->where('flags', 'like', '%"' . $flags . '"%');

        $media_uses = $query
            ->with('media')
            ->get();

        return \response([
            'media' => $media_uses->pluck('media_id'),
            'media_uses' => $media_uses
                ->map(fn(MediaUse $mediaUse) => new MediaEntryResource($mediaUse)),
        ]);
    }


    public function saveMedia(
        Request $request,
        CarConfiguratorModel $carConfigurator,
        CarConfiguratorRepository $repository
    ): Response {
        $flags = collect($request->input('tags'))->map(fn($item) => Str::slug($item))->implode('-');
        $repository->saveMedia($carConfigurator, $flags, $request->input('media'), $request->input('wheel_images', []));
        return response('ok');
    }

    public function importMedia(
        Request $request,
        CarConfiguratorModel $carConfigurator,
        CarConfiguratorRepository $repository
    ): Response {
        $repository->importMedia($carConfigurator, $request->file('file'));
        return response('ok');
    }
}
