<?php

namespace App\Crm;

use App\Contracts\AddsDealershipFields;
use App\Facades\Settings;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Mail\NewEnquiryMail;

class SendToRapidRTC implements EnquiryAction, AddsDealershipFields
{
    use CheckForExpiredAction;

    private EnquiryModel $enquiry;
    private EnquiryActionModel $action;

    public function enabled(): bool
    {
        return Settings::get('crm-rapid-rtc-enabled') === true;
    }

    /**
     * Data fields to add to form
     *
     * @return array
     */
    public function formAdditionalDataFields(): array
    {
        return [
            'fallback-recipient' => [
                'type' => 'text',
                'validation' => [
                    'required',
                    'email',
                ],
                'label' => 'Keyloop Leads fallback email (if not found from dealer)'
            ],
        ];
    }

    /**
     * Data fields to add to questions
     *
     * @return array[]
     */
    public function formQuestionDataFields(): array
    {
        return [];
    }

    /**
     * Data fields to add to dealership
     *
     * @return array
     */
    public function dealershipAdditionalDataFields(): array
    {
        return [
            'rapid-rtc-mailbox' => [
                'type' => 'text',
                'validation' => [
                    'required',
                    'email',
                ],
                'label' => 'Keyloop Leads email address'
            ],
        ];
    }

    /**
     * Handle the action
     *
     * @param EnquiryActionModel $action
     * @param EnquiryModel $enquiry
     * @return bool
     */
    public function handle(EnquiryActionModel $action, EnquiryModel $enquiry): ?bool
    {
        $this->enquiry = $enquiry;
        $this->action = $action;

        if ($this->expired($action)) {
            return null;
        }

        $recipient = $this->getRecipient();
        if (!empty($recipient)) {
            $class = config('crm.new_enquiry_email');
            Mail::to($recipient)->send(new $class($enquiry));
            return true;
        }
        return false;
    }

    /**
     * Message when successfully processed
     *
     * @return string
     */
    public function successDetails(): string
    {
        return "Sent to " . $this->getRecipient();
    }

    /**
     * Message when failed to process
     *
     * @return string
     */
    public function failureReason(): string
    {
        if (empty($this->getRecipient())) {
            return 'no recipient specified';
        }
        return 'issue sending mail';
    }

    /**
     * Find recipient email
     *
     * @return string
     */
    private function getRecipient(): string
    {
        $recipient = '';
        if (in_array($this->enquiry->reason_type, ['vehicle', 'offer'])) {
            $recipient = $this->enquiry->reason->dealership->data['rapid-rtc-mailbox'] ?? '';
        }

        if (empty($recipient)) {
            $recipient = $this->action->data['fallback-recipient'] ?? '';
        }
        return $recipient;
    }
}
