<?php

namespace Mtc\AutoTraderStock\Services;

use Carbon\Carbon;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\AutoTraderStock\Contracts\AutoTraderServiceContract;

class AutoTraderService implements AutoTraderServiceContract
{
    protected const PAGE_SIZE = 50;

    /**
     * @var AutoTraderApi
     */
    protected $api;

    /**
     * @var string
     */
    protected $merchant_id;

    public function __construct($merchant_id, AutoTraderApi $api)
    {
        $this->api = $api;
        $this->merchant_id = $merchant_id;
    }

    public function fetchVehicles($quick_sync = false): Collection
    {
        $params = [
            'pageSize' => self::PAGE_SIZE,
            'advertiserId' => $this->merchant_id,
            'features' => 'true',
        ];
        if ($quick_sync) {
            $params['minLastUpdatedByAdvertiser'] = Carbon::now()->subMinutes(15)->format('Y-m-d\TH:i:s\Z');
        }

        try {
            $data = $this->api->get("stock?" . $this->buildParams($params));
            if (empty($data['results'])) {
                Log::warning("API response has no results for {$this->merchant_id}", $data);
                return collect();
            }
        } catch (ClientException $exception) {
            Log::error($exception->getMessage(), [
                'exception' => $exception,
                'response_headers' => $exception->getResponse()->getHeaders()
            ]);
            return collect([]);
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [ $exception]);
            return collect([]);
        }

        $vehicles = $data['results'];
        if ($data['totalResults'] > self::PAGE_SIZE) {
            $page_count = ceil($data['totalResults'] / self::PAGE_SIZE);
            for ($page = 2; $page <= $page_count; $page++) {
                try {
                    $params['page'] = $page;
                    $url = "stock?" . $this->buildParams($params);
                    $data = $this->api->get($url);
                    $vehicles = array_merge($vehicles, $data['results']);
                } catch (ClientException $exception) {
                    Log::error($exception->getMessage(), [
                        'exception' => $exception,
                        'response_headers' => $exception->getResponse()->getHeaders()
                    ]);
                } catch (\Exception $exception) {
                    Log::error($exception->getMessage(), [ $exception]);
                }
            }
        }
        return collect($vehicles);
    }

    public function fetchAdditional(array $vehicle_details): array
    {
        $params = [
            'derivativeId' => $vehicle_details['vehicle']['derivativeId'],
            'effectiveDate' => $vehicle_details['vehicle']['firstRegistrationDate'] ?? Carbon::now()->format('Y-m-d'),
            'advertiserId' => $this->merchant_id,
        ];
        $feature_url = "taxonomy/features?" . $this->buildParams($params);
        try {
            return [
                'features' => $this->api->get($feature_url)['features'] ?? []
            ];
        } catch (ClientException $exception) {
            Log::error($exception->getMessage(), [
                'exception' => $exception,
                'response_headers' => $exception->getResponse()->getHeaders()
            ]);
            return [];
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [ $exception]);
            return [];
        }
    }

    protected function buildParams(array $params): string
    {
        return collect($params)
            ->map(function ($value, $key) {
                return $key . '=' . $value;
            })->implode('&');
    }
}
