<?php

namespace Mtc\AutoTraderStock\Imports;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Mtc\AutoTraderStock\Contracts\AutoTraderApiImportContract;
use Mtc\Core\Facades\Media;
use Mtc\Core\Images\ImageUploader;
use Mtc\Shop\Item;

class AutoTraderApiToShopItemImport extends AutoTraderApiImportContract
{
    private $image_uploader;

    public function __construct(ImageUploader $image_uploader)
    {
        $this->image_uploader = $image_uploader;
        $this->image_uploader->setFolders(Media::getFolders('product_folders'));
    }

    public function exists(array $vehicle_data): bool
    {
        return Item::query()->where('epos_code', $vehicle_data['metadata']['stockId'])->exists();
    }

    public function update(array $vehicle_data)
    {
        $vehicle = Item::query()->where('epos_code', $vehicle_data['metadata']['stockId'])->firstOrFail();

        $vehicle->fill([
           'deleted' => $this->isVehicleDeleted($vehicle_data) ? 1 : 0,
           'price' => $vehicle_data['adverts']['retailAdverts']['price']['amountGBP'],
        ]);

        $vehicle->save();
    }

    public function add(array $vehicle_data)
    {
        /** @var Item $vehicle */
        $vehicle = Item::query()->create($this->mapDataToColumns($vehicle_data));

        $this->importImages($vehicle, $vehicle_data['media']['images']);
        if (!empty($vehicle_data['enriched_data']['features'])) {
            $this->importFeatures($vehicle, $vehicle_data['enriched_data']['features']);
        }
    }

    public function removeOld(Collection $vehicles)
    {
        Item::query()
            ->whereNotIn('epos_code', $vehicles->pluck('metadata.stockId'))
            ->update([
                'deleted' => 1,
            ]);
    }

    protected function mapDataToColumns($vehicle_data): array
    {
        if (!empty($vehicle_data['vehicle']['firstRegistrationDate'])) {
            $date_of_registration = Carbon::createFromFormat(
                'Y-m-d',
                $vehicle_data['vehicle']['firstRegistrationDate']
            );
        } else {
            $date_of_registration = null;
        }

        $sale_price = $price = $vehicle_data['adverts']['retailAdverts']['price']['amountGBP'];
        if (!empty($vehicle_data['adverts']['retailAdverts']['manufacturerRRP']['amountGBP'])) {
            $price = $vehicle_data['adverts']['retailAdverts']['manufacturerRRP']['amountGBP'];
        } else {
            $sale_price = 0;
        }

        return [
            'epos_code' => $vehicle_data['metadata']['stockId'],
            'is_new' => $vehicle_data['vehicle']['ownershipCondition'] === 'New',
            'registration_number' => $vehicle_data['vehicle']['registration'],
            'make' => $vehicle_data['vehicle']['make'],
            'model' => $vehicle_data['vehicle']['model'],
            'name' => $vehicle_data['vehicle']['model'],
            'derivative' => $vehicle_data['vehicle']['derivative'],
            'autotrader_id' => $vehicle_data['vehicle']['derivativeId'],
            'engine_size' => $vehicle_data['vehicle']['engineCapacityCC'],
            'engine_size_unit' => 'cc',
            'fuel_type' => $vehicle_data['vehicle']['fuelType'],
            'year' => $vehicle_data['vehicle']['yearOfManufacture'],
            'mileage' => $vehicle_data['vehicle']['odometerReadingMiles'],
            'mileage_unit' => 'Mi',
            'body_style' => $vehicle_data['vehicle']['bodyType'],
            'colour' => $vehicle_data['vehicle']['colour'],
            'transmission' => $vehicle_data['vehicle']['transmissionType'],
            'doors' => $vehicle_data['vehicle']['doors'],
            'previous_owners' => $vehicle_data['vehicle']['previousOwners'],
            'price' => $price,
            'sale_price' => $sale_price,
            'sort_price' => $price,
            'price_exvat' => $price / 1.2,
            'price_extra' => $price,
            'wheel_base_type' => $vehicle_data['vehicle']['wheelbaseType'],
            'cab_type' => $vehicle_data['vehicle']['cabType'],
            'style' => $vehicle_data['vehicle']['style'],
            'sub_style' => $vehicle_data['vehicle']['subStyle'],
            'number_of_berths' => $vehicle_data['vehicle']['berths'],
            'axle' => $vehicle_data['vehicle']['axleConfiguration'],
            'video_url' => $vehicle_data['media']['video']['href'],
            'date_of_registration' => $date_of_registration,
            'attention_grabber' => $vehicle_data['adverts']['retailAdverts']['attentionGrabber'],
            'description' => $vehicle_data['adverts']['retailAdverts']['description'] . ' '
                . $vehicle_data['adverts']['retailAdverts']['description2'],
            'vin' => $vehicle_data['vehicle']['vin'],
            'vat_rate' => config('settings.VAT_RATE'),
            'deleted' => $this->isVehicleDeleted($vehicle_data) ? 1 : 0,
        ];
    }

    protected function importImages(Item $vehicle, $image_list)
    {
        collect($image_list)
            ->each(function ($image_url, $index) use ($vehicle) {
                $image_url = str_replace('{resize}', '', $image_url['href']);
                try {
                    $file_path = $this->image_uploader->uploadFromUrl($image_url);
                    if (!empty($file_path['error'])) {
                        return;
                    }
                    $vehicle->images()
                        ->create([
                            'name' => $file_path['name'],
                            'default' => $index == 0 ? 1 : 0,
                        ]);
                } catch (\Exception $exception) {
                    Log::warning('Failed to import image', [
                        'vehicle_id' => $vehicle->id,
                        'image_url' => $image_url,
                        'error' => $exception,
                    ]);
                }
            });
    }

    public function importFeatures(Item $item, $feature_list)
    {
        $item->features()->delete();
        collect($feature_list)
            ->each(function ($feature) use ($item) {
                $item->features()
                    ->create([
                        'name' => $feature['name'],
                        'type' => $feature['type'],
                        'category' => $feature['category'],
                        'price' => $feature['basicPrice'],
                    ]);
            });
    }

    protected function isVehicleDeleted($vehicle_data): bool
    {
        $deleted_states = [
            'DELETED',
            'WASTEBIN',
        ];

        return in_array($vehicle_data['metadata']['lifecycleState'], $deleted_states);
    }
}
