<?php

namespace Mtc\Auction\Lot;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Lot;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Events\BasketItemDeleted;
use Mtc\Basket\Events\BasketItemDeleting;
use Mtc\Basket\Events\BasketItemSaving;
use Mtc\Basket\Events\BasketSaved;
use Mtc\Basket\Events\BasketSaving;
use Mtc\Money\Price;
use Mtc\Orders\Events\OrderPaid;
use Mtc\ShippingManager\Events\SetDeliveryForBasket;

class LotLatePaymentFee extends Model
{

    protected $table = 'auction_lot_late_fees';

    public $days_charged;

    protected $fillable = [
        'days_charged'
    ];


    public function lot()
    {
        return $this->belongsTo(Lot::class, 'lot_id');
    }

    /**
     * Show price attribute surcharge
     *
     * @return mixed
     */
    public function getPriceAttribute()
    {
        //Price is considered to be VAT inclusive and vat being added to the whole charge not per day
        $late_payment_fee_percentage = config('auction.settings.default_late_payment_fee_percentage');

        $price = ($this->lot->hammer_price / 100 * $late_payment_fee_percentage) * $this->lot->ends_at->diff(
                Carbon::now()
            )->days;

        $price = new Price($price/*, $this->taxAttribute()*/);
        $price->calculate();

        return $price;
    }

    /**
     * Show name attribute from surcharge
     *
     * @return mixed
     */
    public function getNameAttribute()
    {
        return $this->lot->name . ' Late Payment Fee';
    }

    /**
     * Show price attribute surcharge
     *
     * @return mixed
     */
    public function getTaxAttribute()
    {
        return $this->surcharge->getPriceAttribute()
                               ->tax();
    }

    public function taxAttribute()
    {
        return 0;
    }

    public static function processOrderPayment(OrderPaid $order_paid)
    {
        $surcharges = collect($order_paid->getOrder()->surcharges);

        if ($surcharges->where('surcharge_type', LotLatePaymentFee::class)
                       ->isNotEmpty()) {
            self::saveDaysChargedStatically($surcharges, $order_paid->getOrder()->id);
        }
    }

    public static function saveDaysChargedStatically($surcharges, $order_id)
    {
        $surcharge_ids = $surcharges->pluck('surcharge_id');

        LotLatePaymentFee::query()
                         ->whereIn('id', $surcharge_ids)
                         ->with('lot')
                         ->get()
                         ->each(function (LotLatePaymentFee $fee) use ($surcharges, $order_id) {
                             $fee->days_charged = $fee->lot->ends_at->diff(Carbon::now())->days;
                             $fee->save();
                         });
    }

    public function saveDaysCharged()
    {
        $this->days_charged = $this->lot->ends_at->diff(Carbon::now())->days;
        $this->save();
    }

    public static function removeAnyNoLongerApplicableLateFees(){
        $basket_interface = App::make(BasketRepositoryInterface::class);
        
        $lots_in_basket = $basket_interface->getModel()->items()->where('purchasable_type', Lot::class)->pluck('purchasable_id')->toArray();
        
        $invalid_late_fee_surcharges = $basket_interface->getModel()->surcharges()->where('surcharge_type', Lot\LotLatePaymentFee::class)->whereHasMorph('surcharge', [Lot\LotLatePaymentFee::class], function($query) use ($lots_in_basket){
          return   $query->whereNotIn('lot_id', $lots_in_basket);
        })->get()->pluck('id')->toArray();

        $basket_interface->getModel()->surcharges()->whereIn('id', $invalid_late_fee_surcharges)->delete();
        
    }
    public static function removeAnyNoLongerApplicableLateFeesOnDeliveryUpdate(SetDeliveryForBasket $setDeliveryForBasket){
        $lots_in_basket = $setDeliveryForBasket->getBasket()->items()->where('purchasable_type', Lot::class)->pluck('purchasable_id')->toArray();
        
        $invalid_late_fee_surcharges = $setDeliveryForBasket->getBasket()->surcharges()->where('surcharge_type', Lot\LotLatePaymentFee::class)->whereHasMorph('surcharge', [Lot\LotLatePaymentFee::class], function($query) use ($lots_in_basket){
          return   $query->whereNotIn('lot_id', $lots_in_basket);
        })->get()->pluck('id')->toArray();
        
        $setDeliveryForBasket->getBasket()->surcharges()->whereIn('id', $invalid_late_fee_surcharges)->delete();
        
    }

}