<?php

namespace Mtc\Auction\Lot;

use Mtc\Orders\Events\OrderPaid;
use Mtc\Auction\Lot;
use Mtc\Auction\Auction;
use Mtc\Auction\Lot\Storage;
use Mtc\Orders\Item;
use Mtc\Orders\Order;
use Carbon\Carbon;

trait HasOrders
{
    /**
     * Relationship with order that was used for registering lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function registerOrder()
    {
        return $this->belongsTo(Order::class, 'register_order_id');
    }
    
    /**
     * Relationship with order that was used to purchase lot
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function buyOrder()
    {
        return $this->belongsTo(Order::class, 'buy_order_id');
    }
    
    public static function processOrderPayment(OrderPaid $order_paid)
    {
        $order_items = collect($order_paid->getOrder()->items);

        if ($order_items->where('purchasable_type', 'Mtc\Auction\Lot\Registration')->isNotEmpty()) {
            self::markAsRegisteredFromOrder($order_items, $order_paid->getOrder()->id);
        }

        if ($order_items->where('purchasable_type', 'Mtc\Auction\Lot')->isNotEmpty()) {
            self::markAsPaidFromOrder($order_items->where('purchasable_type', 'Mtc\Auction\Lot'), $order_paid->getOrder()->id, $order_paid->getOrder()->member_id);
        }
        
        if ($order_paid->getOrder()->deliverySurcharge->surcharge_id == config('auction.storage.delivery_id')) {
            self::setLotsInStorage($order_items->where('purchasable_type', 'Mtc\Auction\Lot'), $order_paid->getOrder()->member_id);
        }

        if ($order_items->where('purchasable_type', 'Mtc\Auction\Lot\Storage')->isNotEmpty()) {
            self::removeLotsFromStorage($order_items->where('purchasable_type', 'Mtc\Auction\LotStorage'));
        }
    }

    /**
     * Mark Lots as registered as payment is confirmed
     *
     * @param $order_item_collection
     */
    protected static function markAsRegisteredFromOrder($order_item_collection, $order_id)
    {
        $size_ids = $order_item_collection->pluck('sizeid');

        Lot::query()
           ->whereIn('id', $size_ids)
           ->get()
           ->each(function (Lot $lot) use ($order_item_collection, $order_id) {
               $does_not_have_reserve_paid = $order_item_collection
                   ->where('sizeid', $lot->id)
                   ->where('purchasable_type', 'Mtc\Auction\Lot\Registration')
                   ->isEmpty();

               if ($does_not_have_reserve_paid) {
                   $lot->reserve_price = 0;
               }

               // TODO: Email about new lots registered with Delivery info
               $lot->register_order_id = $order_id;
               $lot->status = Status::REGISTERED;
               $lot->save();
           });
    }

    /**
     * Mark lots as paid
     *
     * @param $order_item_collection
     */
    protected static function markAsPaidFromOrder($order_item_collection, $order_id, $member_id)
    {
        $order_item_collection->each(function(Item $line_item) use ($order_id, $member_id){
            if($line_item->purchasable_type == 'Mtc\Auction\Lot'){
                $lot = Lot::find($line_item->purchasable_id);
                if ($lot->is_buy_it_now) {
                    $lot->buyer_id = $member_id;
                }
                $lot->status = Status::PAID;
                $lot->buy_order_id = $order_id;
                $lot->save();
            }
        });
    }

    /**
     * Mark lots as in storage
     *
     * @param $order_item_collection
     */
    protected static function setLotsInStorage($order_item_collection, $member_id)
    {
        $order_item_collection->each(function(Item $line_item) use ($member_id){
            if($line_item->purchasable_type == 'Mtc\Auction\Lot'){
                $lot = Lot::find($line_item->purchasable_id);
                Storage::query()
                       ->create([
                           'auction_id' => $lot->auction_id,
                           'lot_id' => $lot->id,
                           'member_id' => $member_id,
                           'storage_expires_at' => Carbon::now()->addDays(config('auction.storage.days_allowance'))
                       ]);
            }
        });
    }

    /**
     * Take lots out from storage
     *
     * @param $order_item_collection
     */
    protected static function removeLotsFromStorage($order_item_collection)
    {
        $order_item_collection->each(function(Item $line_item){
            if($line_item->purchasable_type == 'Mtc\Auction\Lot\Storage'){
                $storage = Storage::find($line_item->purchasable_id);
                $lot = $storage->lot;
                Storage::query()
                       ->where('lot_id', $lot->id)
                       ->delete();
            }
        });
    }
}