<?php

namespace Mtc\Auction\Lot;

use Carbon\Carbon;
use Mtc\Auction\Auction;
use Mtc\Auction\Lot\ComplementaryPriority;
use Mtc\Auction\Lot\Status;

/**
 * Trait HasComplementary
 *
 * @package MTC\Modules\Auction\Classes\Lot
 */
trait HasComplementary
{

    /**
     * Returns list of similar items
     *
     * @return array list of complementary items
     */
    public function getComplementary()
    {
        $items = collect([]);
        $outstanding = config('auction.max_complementary_lots');
        $complementary_rules = ComplementaryPriority::prioritiesAsDatabaseFieldList();

        if ($this->auction->admin_type === 'upcoming') {
            return $items;
        }

        if ($current_auction = Auction::currentLiveAuction()) {
            while ($outstanding <= config('auction.max_complementary_lots') && $complementary_rules->isNotEmpty()) {
                $items = $items->merge($this->getSimilarLots($complementary_rules, $items->pluck('id'), $outstanding, $current_auction->id));
                $outstanding = config('auction.max_complementary_lots') - $items->count();
                $complementary_rules->pop();
            }
        }

        $complementary_rules = ComplementaryPriority::prioritiesAsDatabaseFieldList();

        while ($outstanding <= config('auction.max_complementary_lots') && $complementary_rules->isNotEmpty()) {
            $items = $items->merge($this->getSimilarLots($complementary_rules, $items->pluck('id'), $outstanding));
            $outstanding = config('auction.max_complementary_lots') - $items->count();
            $complementary_rules->pop();
        }

        if ($items->count() < config('auction.max_complementary_lots')) {
            $items = $items->merge($this->getRandomLots($items->pluck('id'), $outstanding));
        }
        return $items
            ->map(function ($lot) {
                $lot->time_left = Carbon::now()->diffInSeconds($lot->ends_at, false);
                return $lot;
            });
    }

    /**
     * Query for finding similar lots based on $match_requirements
     *
     * @param $match_requirements
     * @param $exclude
     * @param $count
     * @return mixed
     */
    protected function getSimilarLots($match_requirements, $exclude, $count, $auction_id = null)
    {
        if (is_null($auction_id)) {
            $auction_id = $this->auction_id;
        }

        return self::query()
            ->where('auction_id', $auction_id)
            ->where('id', '!=', $this->id)
            ->whereNotIn('id', $exclude)
            ->whereHas('custom', function ($custom_query) use ($match_requirements) {
                $match_requirements->each(function ($rule) use ($custom_query) {
                    $custom_query->where($rule, $this->custom[$rule]);
                });

                return $custom_query;
            })
            ->whereIn('status', array_merge([Status::LIVE], Status::$past_statuses))
            ->inRandomOrder()
            ->take($count)
            ->get();

    }

    /**
     * Hail Mary for getting complementary lots - random lots from this auction
     *
     * @param $exclude
     * @param $count
     * @return mixed
     */
    protected function getRandomLots($exclude, $count, $auction_id = null)
    {
        if (is_null($auction_id)) {
            $auction_id = $this->auction_id;
        }

        return self::query()
            ->where('auction_id', $auction_id)
            ->where('id', '!=', $this->id)
            ->whereNotIn('id', $exclude)
            ->whereIn('status', array_merge([Status::LIVE], Status::$past_statuses))
            ->take($count)
            ->get();

    }
}