<?php

namespace Mtc\Auction\Http\Controllers;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Crypt;
use Mtc\Auction\Auction;
use Mtc\Auction\AuctionMember;
use Mtc\Auction\CollectionPoint;
use Mtc\Auction\Lot;
use Mtc\Auction\PickupMethod;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\HasMembersNavigation;

class LotRegistrationController extends Controller
{
    use HasMembersNavigation;
    /**
     * View Lot
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function index(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        if (is_null($this->member) || !$this->member->exists) {
            return redirect()->to('/members/login');
        }
        
        $this->member = AuctionMember::mutateMemberObject($this->member);

        $next_auctions = Auction::query()
                                ->where('submit_deadline', '>', Carbon::now())
                                ->orderBy('starts_at', 'asc')
                                ->take(2)
                                ->get();

        $collection_points = CollectionPoint::query()
                                            ->where('active', true)
                                            ->orderBy('order')
                                            ->orderBy('id')
                                            ->get();

        $auction_member = AuctionMember::mutateMemberObject($this->member);

        app('twig')->addGlobal("is_members_area", true);

        $auction_member->can_sell = true;
        if(config('auction.settings.registration_fee_for_sellers')) {
            $auction_member->can_sell = $auction_member->auction_fields->seller_membership_fee_paid;
        }

        if ($request->wantsJson()) {
            return json_encode([
                                   'lot_sending_methods' => PickupMethod::all(),
                                   'payout_methods'      => AuctionMember::$payout_methods,
                                   'next_auction'        => $next_auctions->shift(),
                                   'fallback_auction'    => $next_auctions->shift(),
                                   'member'              => $auction_member,
                                   'collection_points'   => $collection_points,
                               ]);
        } else {
            $this->page_meta['page_title'] = "Sell Your Lot";
            $this->page_meta['title']      = "Sell Your Lot | " . config('app.name');
            return template('Auction/selling/register.twig', [
                'page_meta'           => $this->page_meta,
                'lot_sending_methods' => PickupMethod::all(),
                'payout_methods'      => AuctionMember::$payout_methods,
                'next_auction'        => $next_auctions->shift(),
                'fallback_auction'    => $next_auctions->shift(),
                'member'              => $auction_member,
                'collection_points'   => $collection_points,
                'sections' => $this->sections(),
            ]);
        }
    }

    /**
     * Process registration request
     *
     * @param Request $request
     *
     * @return array
     */
    public function store(Request $request)
    {
        $this->member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;

        if (!$this->member->exists) {
            return redirect()->to('/members/login');
        }

        if ($request->input('step')) {
            return $this->validateIndividualSteps($request);
        }

        $validated = $this->validate($request, [
            'items'                  => 'required',
            'items.*.name'           => 'required',
            'items.*.reserve_price'  => 'numeric|required_if:items.*.has_reserve_price,1',
            'items.*.info'           => 'present',
            'delivery_method_id'     => 'required',
            'delivery_method_date'   => 'present|required_if:delivery_method_id,1,2',
            //'collection_point_name' => 'present|required_if:delivery_method_id,1,2', // TODO: This should only be expected in "has collection points"
            'payment_method'         => 'required',
            'payment_method_details' => 'present',
        ]);

        $register_price                 = config('auction.fees.registration.default_value');
        $reserve_price                  = config('auction.fees.reserve.default_value');
        
        $seller_commission_percentage                = config('auction.settings.default_buyer_commission_percentage');
        $buyer_commission_percentage                  = config('auction.settings.default_seller_commission_percentage');
        
        $add_reserve_fee_to_basket      = false;
        $add_registration_fee_to_basket = false;
        if (

            $reserve_price > 0
            && !config('auction.lots.defer_lot_reserve_fees')) {
            $add_reserve_fee_to_basket = true;
        }

        if (

            $register_price > 0
            && !config('auction.lots.defer_lot_registration_fees')) {
            //Add Fee purchasable to basket
            $add_registration_fee_to_basket = true;
        }

        $validated['payment_method_details']['method'] = $validated['payment_method'];
        //Encrypt payment details
        $payout_details = Crypt::encryptString(json_encode($validated['payment_method_details']));

        $next_auction        = Auction::nextAssignableAuction();
        $sellers_form_lot_id = null;
        $sellers_id          = $this->member->id;
        $lot_list            = collect();

        $basket_interface = App::make(BasketRepositoryInterface::class);

        collect($validated['items'])->each(
                function ($item) use (
                    $validated,
                    $basket_interface,
                    $payout_details,
                    $next_auction,
                    $register_price,
                    $reserve_price,
                    &$sellers_form_lot_id,
                    $lot_list,
                    $sellers_id,
                    $add_reserve_fee_to_basket,
                    $add_registration_fee_to_basket,
                    $seller_commission_percentage,
                    $buyer_commission_percentage
                ) {

                    $delivery_creation_array = [
                        'delivery_method_id'    => (int)$validated['delivery_method_id'],
                        'delivery_method_date'  => $validated['delivery_method_date'] ?: null,
                    ];

                    $lot = Lot::query()
                              ->create([
                                           'published'                    => 1,
                                           'name'                         => $item['name'],
                                           'auction_id'                   => (!is_null(
                                               $next_auction
                                           )) ? $next_auction->id : null,
                                           'ends_at'                      => (!is_null(
                                               $next_auction
                                           )) ? $next_auction->ends_at : null,
                                           'status'                       => (!empty($register_price)
                                                                              || !empty($reserve_price)) ? Lot\Status::NOT_PAID : Lot\Status::SUBMITTED_ONLINE,
                                           'weight'                       => 2,
                                           'seller_id'                    => $sellers_id,
                                           'description'                  => $item['info'],
                                           'seller_commission_percentage' => $seller_commission_percentage,
                                           'buyer_commission_percentage'  => $buyer_commission_percentage,
                                           'reserve_price'                => $item['has_reserve_price'] ? $item['reserve_price'] : 0,
                                           'relisting_enabled'                => $item['automatic_relist'] ?? 0,
                                       ]);
                    //Create Custom
                    if (is_null($lot->custom)) {
                        $lot->custom()
                            ->create([]);
                    }

                    $lot_list->push($lot);
                    
                    if(array_key_exists('collection_point_id', $validated)){
                        $delivery_creation_array['collection_point_id'] = $validated['collection_point_id'];
                    }
                    
                    
                    //Create Delivery Info
                    $lot->seller_delivery()
                        ->create($delivery_creation_array);
                    
                    //Create Payout Details
                    $lot->seller_payout()
                        ->create([
                                     'payout_details' => $payout_details,
                                 ]);
                    
                    

                    if ($register_price > 0 && $add_registration_fee_to_basket) {
                        //add the item to the basket
                    } elseif ($register_price > 0) {
                        $lot->fees()
                            ->create([
                                         'amount'      => $register_price,
                                         'description' => 'Lot Registration Fee',
                                         'PLU'         => config('auction.lots.purchase_plu_defaults.LOT_REGISTER_PLU'),
                                     ]);
                    }
                    if ($reserve_price > 0 && $add_reserve_fee_to_basket) {
                        //add the item to the basket
                    } elseif ($reserve_price > 0) {
                        $lot->fees()
                            ->create([
                                         'amount'      => $reserve_price,
                                         'description' => 'Lot Reserve Fee',
                                         'PLU'         => config('auction.lots.purchase_plu_defaults.LOT_RESERVE_PLU'),
                                     ]);
                    }
                }
            );

        if ($lot_list->count() > 0) {
            $first_lot = $lot_list->first();
        }

        $seller = $this->member;
        $subject = config('app.name') . " - Lots Submitted";
        $mail = template('Auction/emails/seller_lots_submitted.twig', [
            'lots' => $lot_list,
            'member' => $seller,
        ]);

        email($seller->email, $subject, $mail);

        if (ENVIRONMENT !== 'staging') {
            //email(CONTACT_EMAIL, $subject, $mail);
        }

        //Add fees to basket if the defer fees are enabled


        if (((!empty($reserve_price) && config('auction.lots.defer_lot_reserve_fees'))
            || (!empty($register_price) && config('auction.lots.defer_lot_registration_fees'))) && false) {
            //TODO - The Defer fees process
        } else {
            session()->flash(
                'success',
                __('seller.sell.lots_registered_for_you_link', [
                    'site'   => config('app.name'),
                    'lot_id' => $sellers_form_lot_id,
                ])
            );

            return [
                'status'   => 'ok',
                'redirect' => '/members/my-lots',
            ];
        }
        
        
    }


    /**
     * @param Request $request
     */
    protected function validateIndividualSteps(Request $request)
    {
        switch ($request->input('step')) {
            case 1:
                $this->validate($request, [
                    'items'                 => 'required',
                    'items.*.name'          => 'required',
                    'items.*.reserve_price' => 'numeric|required_if:items.*.has_reserve_price,1',
                    'items.*.info'          => 'present',
                ]);
                break;
            case 2:
                $this->validate($request, [
                    'delivery_method_id'   => 'required',
                    'delivery_method_date' => 'present|required_if:delivery_method_id,1,2',
                    //'collection_point_name' => 'present|required_if:delivery_method_id,1,2',
                ]);
                break;
        }

        return;
    }
}