<?php

namespace Mtc\Auction\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Auction\PriceTrend;
use Mtc\Auction\PriceTrendEntry;
use Mtc\Core\PaginationTemplate;

class PriceTrendController extends Controller
{
    protected $allowed_import_mime_types = [
        'text/plain',
        'text/csv',
        'application/vnd.ms-excel',
    ];
    
    public function index(Request $request)
    {
        $per_page = 50;

        $this->page_meta['title'] = 'Price Trends';

        $price_trends = PriceTrend::query()
                                  //->latest('auction_date')
                                  ->orderBy('catalogued_lot_name')
                                  ->orderBy('id', 'desc');


        $pagination = new PaginationTemplate([
            'item_count'    => $price_trends->count(),
            'per_page'      => $per_page,
            'active_page'   => request()->input('page', 1),
            'show_view_all' => false,
            'page_url'      => '/admin/price-trends/',
        ]);

        return template('Auction/admin/price_trends/index.twig', [
            'price_trends' => $price_trends->take($per_page)
                                           ->offset($per_page * (request()->input('page', 1) - 1))
                                           ->get(),

            'pagination' => $pagination->render(App::make('twig')),
        ]);
    }

    /**
     * View Price Trend Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Price Trend';
        return template('Auction/admin/price_trends/edit.twig', [
            'auction' => new PriceTrend,
            'price_trend_sources' => array_merge(['' => 'Select a source'], []),
            'page_meta' => $this->page_meta,
        ]);
    }
    
    /**
     * View Price Trend Create form
     *
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Syntax
     */
    public function edit($price_trend_id)
    {
        $this->page_meta['title'] = 'Edit Price Trend Info';
        return template('Auction/admin/price_trends/edit.twig', [
            'price_trend' => PriceTrend::query()->findOrFail($price_trend_id),
            'price_trend_sources' => array_merge(['' => 'Select a source'], []),
            'page_meta' => $this->page_meta,
        ]);
    }
    
    public function store(Request $request){
        $validated = $this->validate($request, [
            'catalogued_lot_name' => 'required',
        ]);

        $price_trend = new PriceTrend;
        $price_trend->fill($validated);
        $price_trend->save();

        session()->flash('success', 'Price Trend Created');
        return redirect()->to("/admin/price-trends/{$price_trend->id}/edit");
    }

    /**
     * Delete a Price Trend
     *
     * @param $price_trend_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($price_trend_id)
    {
        $price_trend = PriceTrend::query()->find($price_trend_id);
        if ($price_trend) {
            $price_trend->delete();
            session()->flash('success', 'Price Trend Removed');
        }
        return redirect()->back();
    }

    /**
     * Process Price Trend import form a supplied CSV file.
     *
     * @param Request $request
     */
    public function import(Request $request)
    {
        ini_set('auto_detect_line_endings', true);
        set_time_limit(60 * 5);
        ini_set('memory_limit', '1G');

        if (get_class($request->file('import_csv')) === 'Illuminate\Http\UploadedFile' && $request->file('import_csv')->isValid()) {
            $import_csv = $request->file('import_csv');

            if (in_array($import_csv->getMimeType(), $this->allowed_import_mime_types)) {
                if (($handle = fopen($import_csv->getRealPath(), 'r')) !== false) {
                    $headings = [];
                    $previous_lot_name = '';

                    while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                        if (substr($data[0], 0, 3) === chr(239) . chr(187) . chr(191)) {
                            $data[0] = substr($data[0], 3);
                        }

                        if (empty($headings)) {
                            if (!empty($data[1])) {
                                $headings = $this->normalizeHeadingRow($data);
                            }

                            continue;
                        } else {
                            $lot_name = $this->normalizeData($data[0]);
                            $highest_bid = 0;
                            $auction_date = '';

                            if (strcasecmp($data[0], $previous_lot_name . ' Count') === 0 || strcasecmp($data[0], 'Grand Count') === 0) {
                                continue;
                            }

                            $previous_lot_name = $data[0];

                            for ($i = 1; $i < count($headings); ++$i) {
                                if (!empty($highest_bid = $this->normalizeData(str_replace(',', '', $data[$i])))) {
                                    $auction_date = $headings[$i];
                                    break;
                                }
                            }

                            if (!empty($highest_bid)) {

                                $price_trend = PriceTrend::query()->where('catalogued_lot_name',$lot_name)->first();
                                
                                if($price_trend){
                                    PriceTrendEntry::create([
                                        'price_trend_id' => $price_trend->id,
                                        'source' => $request->input('source'),
                                        'auction_date' => $auction_date,
                                        'amount' => $highest_bid,
                                    ]);
                                } else {
                                    $price_trend = PriceTrend::create([
                                        'catalogued_lot_name' => $lot_name,
                                    ]);
                                    PriceTrendEntry::create([
                                        'price_trend_id' => $price_trend->id,
                                        'source' => $request->input('source'),
                                        'auction_date' => $auction_date,
                                        'amount' => $highest_bid,
                                    ]);
                                }
                                
                            }
                        }
                    }

                    fclose($handle);

                    session()->flash('success', 'Price Trends imported successfully');
                    return redirect()->back();
                } else {
                    session()->flash('error', 'Price Trends import failed: file type not readable');
                    return redirect()->back();
                }
            } else {
                session()->flash('error', 'Price Trends import failed: file type not allowed');
                return redirect()->back();
            }
        } else {
            session()->flash('error', 'Price Trends import failed');
            return redirect()->back();
        }
    }

    private function normalizeData($string)
    {
        // Remove pound symbol
        $string = str_replace(chr(194) . chr(163), '', $string);
        $string = str_replace(chr(163), '', $string);

        return trim($string);
    }

    private function normalizeHeadingRow($heading)
    {
        $date = new Carbon();

        $heading = array_map(function ($item) use ($date) {
            switch ($item) {
                case 'Bottle Name':
                case 'Name':
                    return 'catalogued_lot_name';
                default:
                    $formats = [
                        'M-y',
                        'M/Y',
                        'F j, Y',
                    ];

                    foreach ($formats as $format) {
                        if ($date->hasFormat($item, $format)) {
                            return Carbon::createFromFormat('Y-m-d', Carbon::createFromFormat($format, $item)->format('Y-m-01'))->format('Y-m-d');
                        }
                    }

                    return $item;
            }
        }, $heading);

        return $heading;
    }
    
}