<?php

namespace Mtc\Auction;

use Illuminate\Support\Str;
use Mtc\Auction\Drivers\ImportAuctionLots;
use Mtc\Auction\Exceptions\DataImporterException;

class DataImporter
{
    /**
     * @var array
     */
    private $importable_types;

    private $allowed_mime_types = [
        'text/csv',
        'text/plain'
    ];
    
    public function __construct($allowed_mime_types = [])
    {
        $this->initialiseImportableTypes();
        $this->setAllowedMimeTypes($allowed_mime_types);
    }

    private function initialiseImportableTypes()
    {
        $this->importable_types = [
            ImportAuctionLots::class,
        ];
    }

    private function getImportable($import_type)
    {
        $import_type = collect($this->importable_types)
            ->filter(function ($importable) use ($import_type) {
                return $importable::getSlug() === $import_type;
            })
            ->first();

        if (empty($import_type)) {
            throw new DataImporterException('Import type not recognised.');
        }

        return new $import_type;
    }

    private function readImportData($import_data)
    {
        $mime_type = '';

        // If this is a data URL we need to decode it first
        if (substr($import_data, 0, 5) === 'data:') {
            preg_match('/^data:([a-z]+\/[a-z0-9\.\-\+]+);base64,(.*)$/', $import_data, $matches);
            $mime_type = $matches[1];
            $import_data = base64_decode($matches[2]);
        }

        if (!in_array($mime_type, $this->allowed_mime_types)) {
            throw new DataImporterException('This file type (' . $mime_type . ') is not allowed. Please try again.');
        }

        // Has BOM - remove it
        if (substr($import_data, 0, 3) === b"\xEF\xBB\xBF") {
            $import_data = substr($import_data, 3);
        }

        $stream = fopen('php://memory', 'r+');
        fwrite($stream, $import_data);
        rewind($stream);

        while ($import_data = fgetcsv($stream)) {
            yield $import_data;
        }
    }

    public function setAllowedMimeTypes($allowed_mime_types)
    {
        $this->allowed_mime_types = (array) $allowed_mime_types;
    }

    /**
     * @param string $import_type
     * @param array  $import_data
     * @param int    $preview_lines 3 lines of data +1 line to account for the headings row
     *
     * @return array
     * @throws Exceptions\DataImporterException
     */
    public function preview($import_type, $import_data, $preview_lines = 3 + 1)
    {
        $importable = $this->getImportable($import_type);
        $import_data_iterator = $this->readImportData($import_data);

        $preview = [
            'available_columns' => $importable->getAvailableColumns(),
            'import_parameters' => collect(),
        ];
        $first_row_completed = false;

        foreach ($import_data_iterator as $import_data_line) {
            if (is_int($preview_lines) && $preview_lines-- <= 0) {
                break;
            }

            if (!$first_row_completed) {
                $column_name = 'A';
                $preview['import_parameters'] = collect($import_data_line)
                    ->map(function ($import_data_column_name) use ($importable, &$column_name) {
                        return [
                            'column_name' => $column_name++ . ': ' . $import_data_column_name,
                            'sample_data' => collect(),
                            'suggested_column' => $importable->getSuggestedColumn($import_data_column_name),
                        ];
                    });

                $first_row_completed = true;
            } else {
                collect($import_data_line)
                    ->each(function ($data, $column_index) use ($preview) {
                        $preview['import_parameters'][$column_index]['sample_data']->push(Str::limit($data, 100, ''));
                    });
            }
        }

        return $preview;
    }

    public function import($import_type, $import_data, $import_parameters)
    {
        $importable = $this->getImportable($import_type);
        $import_data_iterator = $this->readImportData($import_data);

        $first_row_skipped = false;

        foreach ($import_data_iterator as $import_data) {
            if (!$first_row_skipped) {
                $first_row_skipped = true;
                continue;
            }

            $import_data = collect($import_data)
                ->mapWithKeys(function ($import_line, $key) use ($import_parameters) {
                    $suggested_column = $import_parameters[$key]['suggested_column'];

                    if (!empty($suggested_column)) {
                        return [$suggested_column => $import_line];
                    }

                    return [];
                })
                ->toArray();

            $importable->import($import_data);
        }
    }

    public static function getTypes()
    {
        return collect((new self)->importable_types)
            ->mapWithKeys(function ($importable) {
                return [$importable::getSlug() => $importable::getName()];
            });
    }
}