<?php

namespace Mtc\Auction\Bid;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Mtc\Auction\Bid;

class Rule extends Model
{

    /**
     * Table name
     *
     * @var string $table
     */
    protected $table = 'auction_bid_rules';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'step_size',
        'min',
        'max'
    ];

    /**
     * No timestamps on this model
     *
     * @var bool $timestamps
     */
    public $timestamps = false;

    /**
     * Find the highest bid we can place with current max
     *
     * @param Bid $new
     * @param int $min_value
     * @return mixed
     */
    public static function findHighestValidBidStep(Bid $new, $min_value = 0)
    {
        $bid_list = self::buildBidList();

        return $bid_list
                   ->where('bid', '<=', $new->max_bid)
                   ->where('bid', '>=', $new->min_allowed_bid)
                   ->where('bid', '>', $min_value)
                   ->last()['bid'] ?? false;
    }

    /**
     * Find the lowest bid we can place with current max
     *
     * @param Bid $new
     * @param int $min_value
     * @param boolean $strictly_over
     * @return mixed
     */
    public static function findLowestValidBidStep(Bid $new, $min_value = 0, $strictly_over = true)
    {
        if (($minimum_pricing_by_law = $new->lot->starting_price - 1) > $min_value) {
            $min_value = $minimum_pricing_by_law;
        }

        $bid_list = self::buildBidList();

        return $bid_list
                   ->where('bid', '<=', $new->max_bid)
                   ->where('bid', '>=', $new->min_allowed_bid)
                   ->where('bid', $strictly_over ? '>' : '>=', $min_value)
                   ->first()['bid'] ?? false;
    }

    /**
     * Find the lowest next bid value
     *
     * @param int $min_value
     * @param Bid $new
     *
     * @return mixed
     */
    public static function findNextBidStep(Bid $new, $min_value = 0)
    {
        if (($minimum_pricing_by_law = $new->lot->starting_price - 1) > $min_value) {
            $min_value = $minimum_pricing_by_law;
        }

        $bid_list = self::buildBidList();

        return $bid_list
                   ->where('bid', '>', $min_value)
                   ->first()['bid'] ?? false;
    }

    /**
     * Find next step_size based on current_bid
     *
     * @param float $current_bid_value
     * @return float
     */
    public static function getStep(float $current_bid_value): float
    {
        $step = 0.0;
        $rules = Rule::query()
            ->select('step_size')
            ->where('min' ,'<=' ,$current_bid_value)
            ->where('max' ,'>=' ,$current_bid_value)
            ->first();
        if($rules->count() > 0)
        {
            $step = (float)$rules->step_size;
        }
        return $step;

    }

    /**
     * We build up a list of all acceptable bid values
     * This makes easier to find bids that should be placed
     * As everything is already known instead of running a calculation each time
     *
     * @return \Illuminate\Support\Collection
     */
    protected static function buildBidList()
    {
        $list = Cache::get('bid_increment_list');

        if (!$list) {
            $list = collect([]);
            self::query()
                ->orderBy('min', 'asc')
                ->get()
                ->each(function (self $rule) use ($list) {
                    $bid = $list->last()['bid'] ?? 0;
                    do {
                        $bid = $bid + $rule->step_size;
                        $list->push(['bid' => $bid]);
                    } while ($rule->max > $bid);
                });

            Cache::put('bid_increment_list', $list, 100);
        }
        return $list;
    }
}