<?php

namespace Mtc\Auction;

use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Model;
use Mtc\Auction\Bid\BidLogic;
use Mtc\Auction\Bid\BidNotifications;
use Mtc\Core\Currency;
use Mtc\Members\Member;
use Mtc\Auction\Lot;

class Bid extends Model
{
    use BidLogic;
    use BidNotifications;
    //use Importable;
    use SoftDeletes;

    /**
     * @var self $previous_bid
     */
    protected $previous_bid;

    /**
     * Lot of this bid
     *
     * @var Lot $lot
     */
    protected $lot;

    /**
     * Table name
     *
     * @var string $table
     */
    protected $table = 'auction_bids';

    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'lot_id',
        'member_id',
        'max_bid',
        'bid',
        'bid_currency',
        'currency',
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = [
        'deleted_at',
    ];


    /**
     * Relationship with Lot of this bid
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function lot()
    {
        return $this->belongsTo(Lot::class);
    }

    /**
     * Relationship with Member of this bid
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(AuctionMember::class);
    }

    /**
     * Try placing a bid  for user
     * @throws \Exception
     */
    public function tryPlacingBid()
    {
        $this->lot = Lot::query()->findOrFail($this->lot_id);
        $previous_bid  = Bid::query()
           ->where('lot_id', $this->lot_id)
           ->latest()
           ->latest('id')
           ->first();
        if(!is_null($previous_bid)) {
            $this->setPreviousBid($previous_bid);
        }

        if ($this->canPlaceBid()) {
            return $this->placeBid();
        }
        throw new \Exception('You must place a valid Bid');
    }
    
    /**
     * Convert bid to base currency for site
     * This ensures all bids are comparable
     *
     * @param $bid_value
     * @param $selected_currency
     * @return float|int
     */
    public static function convertToBaseCurrency($bid_value, $selected_currency)
    {
        if ($selected_currency == Currency::SITE_CURRENCY) {
            return $bid_value;
        }

        $currency = Currency::query()
                            ->where('currency', $selected_currency)
                            ->firstOrFail();

        return $bid_value / $currency->ratio;
    }

    /**
     * Make sure we save the values in DB with currency information
     */
    protected function setBidValueInCurrency()
    {
        $this->currency = $_SESSION['selected_currency'];

        if ($_SESSION['selected_currency'] == Currency::SITE_CURRENCY) {
            $this->bid_currency = $this->bid;
        }

        $currency = Currency::query()
                            ->where('currency', $_SESSION['selected_currency'])
                            ->firstOrFail();

        $this->bid_currency = $this->bid * $currency->ratio;
    }

    /**
     * Try placing a bid  for user
     * @throws \Exception
     */
    public function validateBid()
    {
        $this->lot = Lot::query()->findOrFail($this->lot_id);
        $this->setPreviousBid(Bid::query()
                                 ->where('lot_id', $this->lot_id)
                                 ->latest()
                                 ->latest('id')
                                 ->first());

        if ($this->canPlaceBid()) {
            return true;
        }
        throw new \Exception('You must place a valid Bid');
    }
    
    /*
     * Never make this recursive
     */
    public function setPreviousBid(Bid $bid){
        $this->previous_bid = $bid;
    }


}