<?php

namespace Mtc\Auction;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Http\Controllers\LotController;
use Mtc\Auction\Member\InteractsWithAuctions;
use Mtc\Auction\Member\MembershipFee;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Core\Admin\Setting;
use Illuminate\Support\Facades\Auth;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Members\Member;
use Mtc\Orders\Events\OrderPaid;

class AuctionMember extends Member
{
    use InteractsWithAuctions;

    /**
     * Make sure auction membership payments are processed and user is updated
     *
     * @param OrderPaidEvent $event
     */
    public static function processOrderPaymentForAuction(OrderPaid $order_paid)
    {

        $order_items = collect($order_paid->getOrder()->items);
        
        $order_member = $order_paid->getOrder()->member;
        
        if(!is_null($order_member)) {
            $auction_member = self::mutateMemberObject($order_member);

            if ($order_items->where('purchasable_type', AuctionMembershipFee::class)
                            ->isNotEmpty()) {
                $order_items->where('purchasable_type', AuctionMembershipFee::class)
                            ->each(function ($line_item) use ($auction_member) {
                                if ($line_item->purchasable->type == 'buyer') {
                                    $auction_member->auction_fields()
                                                   ->update(['buyer_membership_fee_paid' => 1]);
                                    $auction_member->auction_fields->save();
                                }
                                if ($line_item->purchasable->type == 'seller') {
                                    $auction_member->auction_fields()
                                                   ->update(['seller_membership_fee_paid' => 1]);
                                    $auction_member->auction_fields->save();
                                }
                            });
            }
        }
        
    }

    //There might be a new version of this

    /**
     * Add ability to get a masked
     *
     * @return HasOne
     */
    public function auction_fields()
    {
        return $this->hasOne(AuctionMemberFields::class, 'member_id', 'id');
    }


    /**
     * Finds out if a user has placed any bids on a particular lot.
     *
     * @param int $lot_id
     *
     * @return bool
     */
    public function userHasBidsOnLot($lot_id)
    {
        return Bid::where('lot_id', $lot_id)
                  ->where('member_id', $this->id)
                  ->exists();
    }

    /**
     * Relationship with lots that have been won but aren't puchased
     *
     * @return mixed
     */
    public function lotsToPurchase()
    {
        return $this->hasMany(Lot::class, 'buyer_id')
                    ->where('status', Lot\Status::SOLD)
                    ->whereNull('buy_order_id');
    }

    /**
     * Get the list of buyers
     * Probably needs a refactor to optimise this section
     *
     * @return mixed
     */
    public static function buyerList()
    {
        $list = self::query();

        /*if (config('auction.settings.registration_fee_for_bidders') === true) {
            $list = $list->whereNotNull('registration_fee_paid_at');
        }*/

        return $list->get()
                    ->keyBy('id')
                    ->map(function ($item) {
                        return $item->name . ' (' . $item->email . ')';
                    });
    }

    /**
     * Get the list of sellers
     * Probably needs a refactor to optimise this section
     *
     * @return mixed
     */
    public static function sellerList()
    {
        $list = self::query();

        /*if (config('auction.settings.registration_fee_for_sellers') === true) {
            $list = $list->whereNotNull('registration_fee_paid_at');
        }*/

        return $list->get()
                    ->keyBy('id')
                    ->map(function ($item) {
                        return $item->name . ' (' . $item->email . ')';
                    });
    }


    /**
     * Get the list of members.
     *
     * @return array
     */
    public static function idNameEmailList()
    {
        return self::query()
                   ->get()
                   ->sortBy('name')
                   ->keyBy('id')
                   ->map(function ($member) {
                       return $member->name . ' [ ' . $member->email . ' ]';
                   });
    }

    public static function mutateMemberObject(Member $member)
    {
        return self::query()
                   ->find($member->id);
    }

    public static function confirmBuyersRegistrationFeeIsPaid()
    {
        $member = class_exists(MemberAuth::class) ? MemberAuth::user() : new \stdClass;
        
        if(!is_null($member)) {
            $auction_member = self::mutateMemberObject($member);

            $basket_interface = App::make(BasketRepositoryInterface::class);
            
            if(is_null($auction_member->auction_fields)){
                $auction_member->auction_fields = $auction_member->auction_fields()->create([]);
            }

            $membership_fee_object = AuctionMembershipFee::query()->where('type', 'buyer')->first();
            
            if (!$auction_member->auction_fields->buyer_membership_fee_paid
                && (config('auction.settings.registration_fee_for_bidders'))) {
                if (!$basket_interface->hasItem($membership_fee_object)) {
                    $basket_interface->addItem($membership_fee_object, 1, true, false);
                }
                return true;
            } else {
                if ($basket_interface->hasItem($membership_fee_object)) {
                    //Also make sure the membership fee is not in the basket
                    $basket_interface->items()
                                     ->where('purchasable_type', AuctionMembershipFee::class)
                                     ->delete();
                }
            }
        }
        return false;
    }


}