<?php

namespace Mtc\Auction;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\Auction\Lot\Status;
use Mtc\Core\Node;
use Mtc\Core\Traits\Nodeable;
use Mtc\Plugins\SiteMenu\Classes\SiteMenu;

class Auction extends Model
{
    use Nodeable;

    protected $appends = ['time_remaining'];
    
    /**
     * Mass assignable attributes
     *
     * @var array $fillable
     */
    protected $fillable = [
        'name',
        'image',
        'banner_image',
        'submit_deadline',
        'content',
        'title',
        'subtitle',
        'is_live',
        'starts_at',
        'ends_at',
    ];

    /**
     * Cast variables to types
     *
     * @var array $casts
     */
    protected $casts = [
        'ends_at' => 'datetime',
        'original_ends_at' => 'datetime',
        'starts_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];


    /**
     * Whether this node allows partial (wild-card) matching (e.g. browse page)
     *
     * @var bool $allow_partial_url_match
     */
    protected bool $allow_partial_url_match = true;
    

    /**
     * Extend class boot method
     */
    protected static function boot()
    {
        parent::boot();

        self::saved(function (self $auction) {
            if ($auction->node()->count() == 0) {
                $auction->node()
                        ->create([
                            'url' => $auction->getUniqueUrl(),
                            'title' => $auction->name,
                            'status' => 'public',
                        ]);
            }
        });
       
        self::updated(function (self $auction) {
            if ($auction->node()
                        ->count() == 0) {
                $auction->node()
                        ->create([
                                     'url'   => $auction->getUniqueUrl(),
                                     'title' => $auction->name,
                                     'status' => 'public',
                                 ]);
            } else {
                $auction->node()
                        ->update([
                                     'url'   => $auction->getUniqueUrl(),
                                     'title' => $auction->name,
                                     'status' => 'public',
                                 ]);
            }
        });

    }
    
    /**
     * Relationship with Lots in Auction
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function lots()
    {
        return $this->hasMany(Lot::class);
    }

    /**
     * Relationship with featured Lots in Auction
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function featuredLots()
    {
        return $this->lots()
                    ->where('is_featured', 1);
    }
    
    /**
     * Filter only auctions that have ended
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopePast(Builder $query)
    {
        return $query->where('ends_at', '<', Carbon::now());
    }  
    
    /**
     * Filter only auctions that are in the future and not started
     *
     * @param Builder $query
     *
     * @return Builder
     */
    public function scopeFuture(Builder $query)
    {
        return $query->where('ends_at', '>', Carbon::now())
                     ->where('is_live', '!=' ,1);
    }

    /**
     * Get the current auction (live or upcoming)
     *
     * @return Model|Auction
     */
    public static function currentAuction()
    {
        $auction = self::currentLiveAuction();
        if (!$auction) {
            return self::nextUpcomingAuction();
        }

        return $auction;
    }

    /**
     * Find the Auction that is currently live
     *
     * @return Model|null|static
     */
    public static function currentLiveAuction()
    {
        return self::query()
                   ->where('is_live', 1)
                   ->where('ends_at', '>', Carbon::now())
                   ->first();
    }
    
    /**
     * Find the next upcoming auction
     *
     * @return Model|null|static
     */
    public static function nextUpcomingAuction()
    {
        return self::query()
                   ->where('starts_at', '>', Carbon::now())
                   ->orderBy('starts_at', 'asc')
                   ->first();
    }

    /**
     * Find the next auction to which lots can be assigned
     *
     * @return Model|null|static
     */
    public static function nextAssignableAuction()
    {
        return self::query()
                   ->where('submit_deadline', '>', Carbon::now())
                   ->orderBy('submit_deadline', 'asc')
                   ->first();
    }

    /**
     * Get the url for previewing auction
     *
     * @return string
     */
    public function getPreviewUrlAttribute()
    {
        return $this->node->url;
    }

    /**
     * Find the Auction that is currently live
     *
     * @return Model|null|static
     */
    public static function auctionDueToStart()
    {
        return self::query()
                   ->where('starts_at', '<=', Carbon::now())
                   ->where('ends_at', '>', Carbon::now())
                   ->where('is_live', 0)
                   ->first();
    }

    /**
     * Check if auction should start
     *
     * @return bool
     */
    public function shouldStart()
    {
        return !$this->is_live
               && $this->starts_at <= Carbon::now()
               && $this->ends_at >= Carbon::now();
    }



    /**
     * Check if "ends soon" condition is met
     *
     * @return bool
     */
    public function endsSoon()
    {
        /**
         * Todo - Base this on a setting/config
         */
        return Carbon::now()->diffInMinutes($this->ends_at) <= 15;
    }

    /**
     * Admin type determines how lots are displayed in admin page
     *
     * @return string
     */
    public function getAdminTypeAttribute()
    {
        if ($this->is_live) {
            return 'live';
        }

        return $this->ends_at < Carbon::now() ? 'past' : 'upcoming';
    }

    /**
     * Auction list in id => name format
     *
     * @return Builder[]|\Illuminate\Database\Eloquent\Collection|\Illuminate\Database\Query\Builder[]|Collection
     */
    public static function idNameList()
    {
        return self::query()
                   ->orderByDesc('starts_at')
                   ->pluck('name', 'id');
    }
    
    /**
     * Make sure we have unique url for auction
     *
     * @return string
     */
    public function getUniqueUrl()
    {
        $url_slug = $this->slugify_string($this->name);
        return Node::urlExists('/' . $url_slug) ? "/{$url_slug}" : '/' . $url_slug;
    }

    public function slugify_string($string){
        return strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-',$string)));
    }
    
    public function regenerateNode(){
        if ($this->node()
                    ->count() == 0) {
            $this->node()
                    ->create([
                                 'url'   => $this->url,
                                 'title' => $this->name,
                                 'status' => 'public',
                                 'visibility' => 'public',
                             ]);
        } else {
            $this->node()
                    ->update([
                                 'url'   => $this->url,
                                 'title' => $this->name,
                                 'status' => 'public',
                                 'visibility' => 'public',
                             ]);
        }
    }

    /**
     * Each auction has a unique url
     *
     * @return mixed
     */
    public function getUrlAttribute()
    {
        if(!is_null($this->node)){
            return $this->node->url;
        } else {
            return null;
        }
        
    }


    /**
     * Generate basic Stats for the Auction. Only to be ran on Auctions that have ended
     *
     * @return array
     */
    public function getAuctionStats(){
        return [];
        /*$total_lots = $this->lots()->count();
        $sold_lots_count = $this->lots()->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->count();
        $unsold_lots_count = $this->lots()->where('status', Status::NOT_SOLD)->count();
        $cancelled_lots_count = $this->lots()->where('status', Status::CANCELLED)->count();
        $collective_highest_bids = $this->lots()->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->sum('highest_bid');
        $auction_order_ids = $this->lots()->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->whereNotNull('buy_order_id')->groupBy(['buy_order_id'])->pluck('buy_order_id');
        $auction_orders = Order::query()->whereIn('id', $auction_order_ids)->count();
        $auction_orders_total = Order::query()->whereIn('id', $auction_order_ids)->sum('cost_total');
        $auction_buyer_commission = $this->lots()->whereNotNull('buy_order_id')->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->sum('buyer_commission_with_vat');
        $auction_buyer_commission_exvat = $this->lots()->whereNotNull('buy_order_id')->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->sum('buyer_commission');
        $auction_orders_total_exvat = Order::query()->whereIn('id', $auction_order_ids)->sum('cost_total_exvat');
        $auction_orders_delivery_total = Order::query()->whereIn('id', $auction_order_ids)->sum('delivery_cost');
        $auction_orders_delivery_total_exvat = $auction_orders_delivery_total/(1+VAT_RATE/100);

        $sold_lots = $this->lots()->whereIn('status', [Status::SOLD, Status::PAID, Status::SHIPPED])->pluck('id')->toArray();

        $sold_lots_reserve_fees = 0;
        $sold_lots_reserve_fees_exvat = 0;
        $sold_lots_listing_fees = 0;
        $sold_lots_listing_fees_exvat = 0;

        if($sold_lots_count > 0){
            $sold_lots_reserve_fees = Fee::query()->whereIn('lot_id',$sold_lots)->where('PLU', 'RESERVE_FEE')->sum('amount');
            $sold_lots_reserve_fees_exvat = $sold_lots_reserve_fees/(1+VAT_RATE/100);
            $sold_lots_listing_fees = Fee::query()->whereIn('lot_id',$sold_lots)->where('PLU', 'LOT_FEE')->sum('amount');
            $sold_lots_listing_fees_exvat = $sold_lots_listing_fees/(1+VAT_RATE/100);
        }

        return [
            'Total Lots' => $total_lots,
            'Sold Lots' => $sold_lots_count,
            'Unsold Lots' => $unsold_lots_count,
            'Cancelled Lots' => $cancelled_lots_count,
            'Total Orders' => $auction_orders,
            'Collective Bid Value' => '&pound;' . $collective_highest_bids,
            'Total Order Cost Incl. VAT' => '&pound;' . number_format($auction_orders_total,2),
            'Total Order Cost Excl. VAT' => '&pound;' . number_format($auction_orders_total_exvat,2),
            'Total Buyers Commission Taken Incl. VAT' => '&pound;' . number_format($auction_buyer_commission,2),
            'Total Buyers Commission Taken Excl. VAT' => '&pound;' . number_format($auction_buyer_commission_exvat,2),
            'Total Delivery Charges Incl. VAT' => '&pound;' . number_format($auction_orders_delivery_total,2),
            'Total Delivery Charges Excl. VAT' => '&pound;' . number_format($auction_orders_delivery_total_exvat,2),
            'Total Reserve Fees (of sold Lots) Incl. VAT' => '&pound;' . number_format($sold_lots_reserve_fees,2),
            'Total Reserve Fees (of sold Lots) Excl. VAT' => '&pound;' . number_format($sold_lots_reserve_fees_exvat,2),
            'Total Listing Fees (of sold Lots) Incl. VAT' => '&pound;' . number_format($sold_lots_listing_fees,2),
            'Total Listing Fees (of sold Lots) Excl. VAT' => '&pound;' . number_format($sold_lots_listing_fees_exvat,2),
        ];*/

    }


    /**
     * Check if auction can be ended
     *
     * @return bool
     */
    public function shouldEndAuction()
    {
        return $this->ends_at <= Carbon::now()
               && $this->lots()->live()->count() == 0;
    }

    /**
     * End Auction
     */
    public function endAuction()
    {
        $this->is_live = 0;
        $this->save();
    }



    /**
     *
     * Finds and returns next auction that is valid for lot submission
     *
     * @access public
     * @return Builder|Model
     */
    public static function nextSubmittableAuction()
    {
        return self::query()
                   ->where('submit_deadline' , '>' , Carbon::now())
                   ->orderBy('starts_at', 'ASC')
                   ->first();
    }

    /**
     *
     * Finds and returns next auction that is hasn't started yet
     *
     * @access public
     * @return Builder|Model
     */
    public static function nextAvailableAuction()
    {
        return self::query()
                   ->where('starts_at' , '>' , Carbon::now())
                   ->orderBy('starts_at', 'ASC')
                   ->first();
    }

    /**
     * Get the time_remaining value
     * This gives seconds until Auction timer expires.
     * Auction timer will depend whether its upcoming or live auction
     *
     * @return int
     */
    public function getTimeRemainingAttribute()
    {
        $start_datetime = $this->starts_at;
        $end_datetime = $this->ends_at;

        if (Carbon::now() < $start_datetime) {
            return Carbon::now()->diffInSeconds($start_datetime);
        }
        return Carbon::now()->diffInSeconds($end_datetime);
    }

    /**
     * Make sure we extend the auction end time if there are lots that are not sold
     */
    public function makeSureAuctionTimerHasNotEnded()
    {
        if (!$this->lots()->live()->exists()) {
            return;
        }
        
        $lot_closest_finish = $this->lots()
                         ->live()
                         ->where('ends_at', '<=', Carbon::now()->addMinute())
                         ->orderByDesc('ends_at')
                         ->first(['ends_at']);
        
        if (is_null($lot_closest_finish)) {
            $end_time = $this->lots()
                             ->live()
                             ->orderBy('ends_at')
                             ->first(['ends_at'])
                ->ends_at;
        } else {
            $end_time = $lot_closest_finish->ends_at;
        }

        $this->ends_at = $end_time;
        $this->save();

    }

    /**
     * Start auction
     */
    public function startAuction()
    {
        $this->is_live = 1;
        $this->save();
        
        $this->toggleActiveAuctionStatusInMenu(1);

        $this->lots()
             ->where('status', Status::READY)
             ->update([
                          'status' => Status::LIVE,
                          'ends_at' => $this->ends_at
                      ]);

        $this->notifyAuctionStarted($this->lots()->where('status', Status::LIVE)->get()->groupBy('seller_id'));
    }

    /**
     * Send notification to sellers that auction has started
     *
     * @param $grouped_lots
     */
    protected function notifyAuctionStarted($grouped_lots)
    {
        $grouped_lots->each(function ($lot_list, $seller_id) {
            $seller = AuctionMember::query()->find($seller_id);
            if (!$seller) {
                return;
            }
            $subject = config('app.name') . " - Auction Now Live";
            $mail = template('Auction/emails/seller_auction_live.twig', [
                'lots' => $lot_list,
                'member' => $seller,
            ]);

            email($seller->email, $subject , $mail);
        });
    }

    /**
     * Get the auction image url
     *
     * @param string $size
     */
    public function imageUrl($size = 'thumbs', $absolute = false)
    {
        $image_folders = App::make('image_folders');
        if (!empty($this->image)) {
            return ($absolute ? SITE_URL : '') . '/' . $image_folders['auction_images'][$size]['path'] . '/' . $this->image;
        }

        $lot = $this->lots()
                    ->whereHas('defaultImage')
                    ->with([
                               'defaultImage'
                           ])
                    ->latest('highest_bid')
                    ->first();

        return ($absolute ? SITE_URL : '') . '/' . $image_folders['lot_images'][$size]['path'] . '/' . $lot->defaultImage->name;

    }
    
    /**
     * Get the auction image url
     *
     * @param string $size
     */
    public function bannerImageUrl($size = 'thumbs', $absolute = false)
    {
        $image_folders = App::make('image_folders');
        if (!empty($this->banner_image)) {
            return ($absolute ? SITE_URL : '') . '/' . $image_folders['auction_images'][$size]['path'] . '/' . $this->banner_image;
        }
        return false;
    }

    /**
     * Update Menu entry for active auction
     *
     * @param $published
     */
    protected function toggleActiveAuctionStatusInMenu($published)
    {
        $menu = SiteMenu::query()->find(config('auction.settings.live_menu_id'));
        if ($menu) {
            $menu->published = $published;
            $menu->save();
        }
    }
}