<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

class CreateAuctionTables extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('auctions', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->text('name');
            $table->text('image')
                  ->nullable();
            $table->longText('content')
                  ->nullable();
            $table->boolean('is_live')
                  ->default('0');
            $table->dateTime('submit_deadline');
            $table->dateTime('starts_at');
            $table->dateTime('ends_at');
            $table->dateTime('original_ends_at');
            $table->timestamps();
        });

        Schema::create('auction_bids', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->integer('member_id')->unsigned();
            $table->decimal('bid');
            $table->decimal('max_bid');
            $table->decimal('bid_currency');
            $table->text('currency');
            $table->timestamps();
            $table->softDeletes();
        });

        Schema::create('auction_bid_rules', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->decimal('step_size');
            $table->decimal('min');
            $table->decimal('max');
            $table->timestamps();
        });

        Schema::create('auction_lots', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('status')
                  ->nullable();
            $table->integer('relisted_from_lot_id')->unsigned()
                  ->nullable();
            $table->integer('auction_id')->unsigned()
                  ->nullable();
            $table->text('reference')
                  ->nullable();
            $table->text('barcode')
                  ->nullable();
            $table->integer('seller_id')->unsigned()
                  ->nullable();
            $table->integer('buyer_id')->unsigned()
                  ->nullable();
            $table->integer('brand_id')->unsigned()
                  ->nullable();
            $table->integer('register_order_id')->unsigned()
                  ->nullable();
            $table->integer('buy_order_id')->unsigned()
                  ->nullable();
            $table->integer('price_trends_id')->unsigned()
                  ->nullable();
            $table->integer('pickup_method_id')->unsigned()
                  ->nullable();
            $table->decimal('highest_bid')
                  ->nullable();
            $table->boolean('published')
                  ->nullable();
            $table->dateTime('ends_at')->nullable();
            $table->dateTime('original_ends_at')->nullable();
            $table->text('name');
            $table->longText('description')
                  ->nullable();
            $table->longText('short_description')
                  ->nullable();
            $table->decimal('weight')
                  ->default('2.00');

            $table->decimal('reserve_price')
                  ->nullable();
            $table->decimal('starting_price')
                  ->nullable();

            $table->decimal('buy_out_price')
                  ->nullable();


            $table->boolean('vat_on_hammer_price')
                  ->nullable();
            $table->boolean('is_featured')
                  ->nullable();
            $table->boolean('is_in_slideshow')
                  ->nullable();

            //The Old System only would store the Highest Bid.
            //Made the hammer price its own value so that it could be edited in the admin area in a more client friendly way
            // And keep data from the live auction but the acting hammer price is always this value
            //And this way there is no complicated maths needed for tax
            $table->decimal('hammer_price')
                  ->nullable();
            $table->decimal('hammer_price_with_vat')
                  ->nullable();
            
            $table->decimal('buyer_commission_percentage')
                  ->nullable();
            $table->decimal('buyer_commission')
                  ->nullable();
            $table->decimal('buyer_commission_with_vat')
                  ->nullable();
            $table->decimal('seller_commission_percentage')
                  ->nullable();
            $table->decimal('seller_commission')
                  ->nullable();
            $table->decimal('seller_commission_with_vat')
                  ->nullable();
            $table->decimal('seller_commission_limit')
                  ->nullable();


            $table->boolean('ignore_late_payment_fee');
            
            $table->integer('num_views');
            $table->timestamp('last_viewed_at');
            
            $table->integer('custom_field_set_id');
            $table->integer('relisting_enabled');

            $table->timestamps();
            $table->softDeletes();
        });

        /** TODO - Lot Submission/Delivery to Auctioneer
         * Original Auction system had fields for how lots would get to the auctioneer
         * Very few clients used this
         * Separating them out so that the main lots table is tidier
         * But we would need to come back to this and revaluate this. 
         * Lot submission as a whole may be redone
         */
        Schema::create('auction_lot_seller_deliveries', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->integer('delivery_method_id')->unsigned();
            $table->date('delivery_method_date');
            $table->text('courier_details')->nullable();
            $table->integer('collection_point_id')->nullable();
            $table->timestamps();
        });
        
        /** TODO - Seller Payout
         * Original Auction system had fields for paying the seller
         * Very few clients used this and may not have worked as intended
         * Separating them out so that the main lots table is tidier
         * We would need to see if this could be automated via Stripe or if it would be a manual process
         * Will need to revisit
         */
        Schema::create('auction_lot_payouts', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->text('payout_reference');
            $table->text('payout_details');
            $table->dateTime('payout_made_at');
            $table->timestamps();
        });
        
        //Price trends in the old system were just stored JSON strings on the lots themselves
        //Making them into a standalone relation where admins assign lots to use the data of a catalogued lot name
        //It will pull in information from that and add to it automatically at auction end
        //Makes some more sense this way, the json idea would require adding the same json string repeatedly to all lots
        //And updating en masse is easier this way
        Schema::create('auction_lot_price_trends', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->text('catalogued_lot_name');
            $table->timestamps();
        });
        Schema::create('auction_lot_price_trend_entries', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('price_trend_id')->unsigned()
                  ->nullable();
            $table->text('data_source');
            $table->decimal('amount');
            $table->dateTime('auction_date');
            $table->timestamps();
        });
        
        Schema::create('auction_lot_fees', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->decimal('amount');
            $table->text('description')
                  ->nullable();
            $table->text('PLU')
                  ->nullable();
            $table->timestamps();
        });
        
        Schema::create('auction_lot_images', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->text('name');
            $table->boolean('default')
                  ->nullable();
            $table->boolean('hover')
                  ->nullable();
            $table->boolean('zoom')
                  ->nullable();
            $table->integer('order')
                  ->nullable();
            $table->timestamps();
        });

        Schema::create('auction_lot_pickup_methods', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->text('name');
            $table->text('description')
                  ->nullable();
            $table->boolean('warning_text')
                  ->nullable();
            $table->boolean('has_date_picker')
                  ->nullable();
            $table->boolean('has_collection_points')
                  ->nullable();
            $table->timestamps();
        });
        
        Schema::create('auction_lot_collection_points', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->boolean('active')
                  ->nullable();
            $table->text('name');
            $table->integer('order')
                  ->nullable();
            $table->timestamps();
        });
        
        Schema::create('auction_lot_post_live_process_queue', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->integer('member_id')->unsigned();
            $table->text('action');
            $table->timestamps();
        });
        
        Schema::create('auction_storage_lots', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('auction_id')->unsigned();
            $table->integer('lot_id')->unsigned();
            $table->integer('member_id')->unsigned();
            $table->integer('location_id')->unsigned();
            $table->date('storage_expires_at');
            $table->timestamps();
            $table->softDeletes();
        });
        
        Schema::create('auction_storage_lot_locations', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->text('name');
            $table->text('description');
            $table->text('address');
            $table->timestamps();
        });
        
        Schema::create('auction_lot_complementary_priority', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('field_id')->unsigned();
            $table->integer('priority')->unsigned();
            $table->timestamps();
        });
        Schema::create('auction_watchlist_lots', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->integer('member_id')->unsigned();
            $table->tinyInteger('notification_enabled');
            $table->timestamps();
        });

        Schema::create('auction_custom_fields', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name');
            $table->string('type')->index();
            $table->integer('custom_field_list_id')->index()->nullable();
            $table->string('database_field')->index();
            $table->boolean('multiple')->index();
            $table->integer('quantity');
            $table->integer('order')->default(0);
            $table->timestamps();
        });

        Schema::create('auction_custom_fields_sets', function (Blueprint $table) {
            $table->increments('id');
            $table->string('name')->index();
            $table->boolean('default')->index();
            $table->timestamps();
        });

        Schema::create('auction_custom_fields_sets_fields', function (Blueprint $table) {
            $table->increments('id');
            $table->integer('set_id')->index();
            $table->integer('custom_field_id')->index();
            $table->boolean('required')->index();
            $table->boolean('lots')->index();
            $table->boolean('variations')->index();
            $table->boolean('filter')->index();
            $table->integer('order')->index();
            $table->timestamps();
        });
        
        Schema::create('auction_lot_custom', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
        });
        Schema::create('auction_lot_custom_multiple', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('lot_id')->unsigned();
            $table->text('field')->nullable();
            $table->text('value')->nullable();
        });

        /** TODO - Lot Custom Fields and why they are not here
         * Original Auction System was more or less hardcoded to only work for alcohol
         * We did one collectable auction house and it was difficult to change
         *
         * Plan would instead now to be to make lot custom fields load in based on a new config setting
         * "Auction Lot Types"
         * Different types will load different tables for their custom data
         * Plan is at when this is ready, we will will launch with Alcohol (Small Lot), Alcohol (Cask/Large Lot), Collectables and the ability for adding more
         * With an example class for showing how to create new types
         */
    
        /** TODO - User Controlled Global Auction Settings
         * Common Request from clients is the ability to change their settings without getting in contact with us
         * This table would allow this but should be a paid for feature
         * Settings here are right now based on the config file and will supercede the config file if it is enabled
         * May be removed later. Especially if I change how these custom settings are handled
         */
        Schema::create('auction_user_override_settings', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->boolean('active')->nullable();
            $table->integer('late_payment_workdays');
            $table->integer('late_payment_fee_percentage');
            $table->integer('storage_days_allowance');
            $table->integer('storage_late_fee_per_day');
            $table->decimal('insurance_percentage');
            $table->decimal('default_buyer_commission_percentage');
            $table->decimal('default_seller_commission_percentage');
            $table->timestamps();
        });
        
        
        Schema::create('auction_member_fields', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('member_id')->unsigned();
            $table->boolean('buyer_membership_fee_paid')->default(0);
            $table->boolean('seller_membership_fee_paid')->default(0);
            $table->text('seller_information')->nullable();
            $table->timestamps();
        });
        
        
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::dropIfExists('auctions');
        Schema::dropIfExists('auction_bids');
        Schema::dropIfExists('auction_bid_rules');
        Schema::dropIfExists('auction_lots');
        Schema::dropIfExists('auction_lot_seller_deliveries');
        Schema::dropIfExists('auction_lot_payouts');
        Schema::dropIfExists('auction_lot_price_trends');
        Schema::dropIfExists('auction_lot_fees');
        Schema::dropIfExists('auction_lot_images');
        Schema::dropIfExists('auction_lot_pickup_methods');
        Schema::dropIfExists('auction_lot_collection_points');
        Schema::dropIfExists('auction_lot_post_live_process_queue');
        Schema::dropIfExists('auction_storage_lots');
        Schema::dropIfExists('auction_user_override_settings');
        Schema::dropIfExists('auction_lot_complementary_priority');
        Schema::dropIfExists('auction_watchlist_lots');
        Schema::dropIfExists('auction_custom_fields');
        Schema::dropIfExists('auction_custom_fields_sets');
        Schema::dropIfExists('auction_custom_fields_sets_fields');
        Schema::dropIfExists('auction_lot_custom');
        Schema::dropIfExists('auction_lot_custom_multiple');
    }
}
