<?php

namespace Mtc\AdminGoogle2fa\Http\Controllers;

use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Mtc\Core\Admin\User;
use PragmaRX\Google2FAQRCode\Google2FA;

class Google2faController
{
    /**
     * Show 2FA Home
     * @return mixed
     */
    public function show2faManage()
    {
        // check if user has mtc role
        if (!Auth::user()->hasRole('mtc')) {
            return redirect()->route('admin.g2fa.settings');
        }

        $this->page_meta['title'] = 'Manage Google 2FA Users';

        return template('admin-google2fa/users.twig', [
            'page_meta' => $this->page_meta,
            'users' => $this->getUsers()
        ]);
    }

    /**
     * Show 2fa Verification
     * @param Google2FA $google2fa
     * @return mixed
     */
    public function show2faVerify(Google2FA $google2fa)
    {
        $this->page_meta['title'] = 'Google 2FA';

        $user = Auth::user();

        $google2fa_url = $this->getGoogle2faUrl($google2fa);

        $user->can_manage_users = $user->hasRole('mtc');

        return template('admin-google2fa/verify.twig', [
            'google2fa_url' => $google2fa_url,
            'page_meta' => $this->page_meta,
            'user' => $user
        ]);
    }

    /**
     * Process 2fa verification
     * @param Request $request
     * @return mixed
     */
    public function process2faVerification(Request $request)
    {
        $this->page_meta['title'] = 'Google 2FA';

        $user = Auth::user();
        $google2fa = new Google2FA();
        $code = $request->input('one_time_password');
        $valid = $google2fa->verifyKey($user->g2fa_secret, $code);

        if ($valid) {
            session(['g2fa_verified' => true]);
            return redirect()->route('admin.index');
        }

        $user->can_manage_users = $user->hasRole('mtc');

        $response["type"] = "error";
        $response["error"] = "Invalid Authentication Code";

        return template('admin-google2fa/verify.twig', [
            'response' => $response,
            'page_meta' => $this->page_meta,
            'user' => $user
        ]);
    }


    /**
     * Show 2fa Settings
     * @param Request $request
     * @return mixed
     */
    public function show2faSettings(Request $request)
    {
        $this->page_meta['title'] = 'Google 2FA Settings';

        $user = Auth::user();

        // Initialise the 2FA class
        $google2fa = new Google2FA();
        $google2fa_url = $this->getGoogle2faUrl($google2fa);

        $user->can_manage_users = $user->hasRole('mtc');

        return template('admin-google2fa/index.twig', [
            'google2fa_url' => $google2fa_url,
            'page_meta' => $this->page_meta,
            'user' => $user
        ]);
    }

    /**
     * Generate 2FA secret key
     * @return \Illuminate\Foundation\Application|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function generate2faSecret()
    {
        $this->page_meta['title'] = 'Generate Google 2FA Secret';

        $user = Auth::user();
        // Initialise the 2FA class
        $google2fa = new Google2FA();

        // Add the secret key to the user data
        $user->g2fa_secret = $google2fa->generateSecretKey();
        $user->save();

        $google2fa_url = $this->getGoogle2faUrl($google2fa);

        // set responses
        $response["type"] = "success";
        $response["success"] = "Secret key generated";

        $user->can_manage_users = $user->hasRole('mtc');

        return template('admin-google2fa/index.twig', [
            'google2fa_url' => $google2fa_url,
            'response' => $response,
            'page_meta' => $this->page_meta,
            'user' => $user
        ]);
    }

    /**
     * Enable 2FA
     * @param Request $request
     * @return mixed
     */
    public function enable2fa(Request $request)
    {
        $this->page_meta['title'] = 'Enable Google 2FA';

        $user = Auth::user();
        $google2fa = new Google2FA();
        $google2fa_url = $this->getGoogle2faUrl($google2fa);


        $code = $request->input('secret');
        $valid = $google2fa->verifyKey($user->g2fa_secret, $code);

        if ($valid) {
            $user->g2fa_enabled = 1;
            $user->save();

            $response["type"] = "success";
            $response["success"] = "2FA Enabled Successfully";
        } else {
            $response["type"] = "error";
            $response["error"] = "Invalid Authentication Code";
        }

        $user->can_manage_users = $user->hasRole('mtc');

        return template('admin-google2fa/index.twig', [
            'google2fa_url' => $google2fa_url,
            'response' => $response,
            'page_meta' => $this->page_meta,
            'user' => $user
        ]);
    }

    /**
     * Disable 2FA
     * @param Request $request
     * @return mixed
     */
    public function disable2fa(Request $request)
    {

        $this->page_meta['title'] = 'Manage Google 2FA Users';

        // check if user has mtc role
        if (!Auth::user()->hasRole('mtc')) {
            return redirect()->route('admin.g2fa.settings');
        }

        $user = User::find($request->input("user"));
        $user->g2fa_enabled = 0;
        $user->save();

        $response["type"] = "success";
        $response["success"] = "Google 2FA disabled for " . $user->name;

        return template('admin-google2fa/users.twig', [
            'response' => $response,
            'page_meta' => $this->page_meta,
            'users' => $this->getUsers()
        ]);
    }

    /**
     * Check QR Code for Authentication
     * @param $google2fa
     * @return string
     */
    public function getGoogle2faUrl($google2fa): string
    {
        $user = Auth::user();

        if (!$user->g2fa_enabled) {
            return $google2fa->getQRCodeInline(
                config("app.name"),
                $user->email,
                $user->g2fa_secret
            );
        }
        return "";
    }

    /**
     * Get Users
     * @return Collection
     */
    public function getUsers(): Collection
    {
        $users = User::query();

        return $users
            ->latest('id')
            ->with('roles')
            ->get();
    }
}
